﻿<?php
require_once __DIR__ . '/config.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Password Generator - Free Secure Password Creation Tool for GoHighLevel Users | GHL Tools</title>
    <meta name="description" content="Generate strong, secure passwords with our free Password Generator. Perfect for GoHighLevel users to create secure passwords for client accounts and system access.">
    <meta name="keywords" content="password generator, secure password, password strength, strong password, GoHighLevel security, password creation tool, secure password generator">
    <meta name="author" content="GHL Tools">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph / Facebook -->
    <meta property="og:type" content="website">
    <meta property="og:url" content="https://ghl-tools.com/password-generator.php">
    <meta property="og:title" content="Password Generator - Free Secure Password Creation Tool for GoHighLevel Users">
    <meta property="og:description" content="Generate strong, secure passwords with our free Password Generator. Perfect for GoHighLevel users to create secure passwords for client accounts and system access.">
    <meta property="og:image" content="https://ghl-tools.com/images/password-generator-og.jpg">
    
    <!-- Twitter -->
    <meta property="twitter:card" content="summary_large_image">
    <meta property="twitter:url" content="https://ghl-tools.com/password-generator.php">
    <meta property="twitter:title" content="Password Generator - Free Secure Password Creation Tool for GoHighLevel Users">
    <meta property="twitter:description" content="Generate strong, secure passwords with our free Password Generator. Perfect for GoHighLevel users to create secure passwords for client accounts and system access.">
    <meta property="twitter:image" content="https://ghl-tools.com/images/password-generator-og.jpg">
    
    <!-- Structured Data -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "Password Generator",
        "description": "Free tool to generate strong, secure passwords for enhanced security and account protection.",
        "url": "https://ghl-tools.com/password-generator.php",
        "applicationCategory": "SecurityApplication",
        "operatingSystem": "Any",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "publisher": {
            "@type": "Organization",
            "name": "GHL Tools",
            "url": "https://ghl-tools.com"
        }
    }
    </script>
    
    <link rel="stylesheet" href="styles.css">
    <link rel="canonical" href="https://ghl-tools.com/password-generator.php">
</head>
<body>
    <?php inc('includes/header.php'); ?>

    <!-- Hero Section -->
    <section class="hero tool-hero">
        <div class="container">
            <nav class="breadcrumb">
                <a href="index.php">Home</a> > <span>Password Generator</span>
            </nav>
            <div class="hero-content">
                <div class="hero-text">
                    <h1 class="hero-title">
                        Secure <span class="highlight">Password Generator</span>
                    </h1>
                    <p class="hero-subtitle">
                        Create strong, secure passwords to protect your GoHighLevel accounts and client data. 
                        Generate cryptographically secure passwords with customizable options.
                    </p>
                    <div class="hero-stats">
                        <div class="stat-item">
                            <span class="stat-number">256-bit</span>
                            <span class="stat-label">Security</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">Customizable</span>
                            <span class="stat-label">Options</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">100%</span>
                            <span class="stat-label">Free Tool</span>
                        </div>
                    </div>
                </div>
                <div class="hero-visual">
                    <div class="hero-image">
                        <div class="floating-card card-1">
                            <span class="card-icon">🔐</span>
                            <span class="card-text">Secure</span>
                        </div>
                        <div class="floating-card card-2">
                            <span class="card-icon">⚡</span>
                            <span class="card-text">Fast</span>
                        </div>
                        <div class="floating-card card-3">
                            <span class="card-icon">🛡️</span>
                            <span class="card-text">Protected</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Section -->
    <section class="tool-section">
        <div class="container">
            <div class="tool-card featured">
                <div class="tool-header">
                    <div class="tool-icon">🔐</div>
                    <h2>Secure Password Generation Tool</h2>
                    <p>Generate strong, secure passwords with customizable options for length, complexity, and character types.</p>
                </div>
                
                <div class="tool-interface">
                    <div class="input-group">
                        <label for="passwordLength">Password Length</label>
                        <input type="range" id="passwordLength" min="8" max="64" value="16" class="range-slider">
                        <span id="lengthValue">16</span> characters
                    </div>
                    
                    <div class="input-group">
                        <label>Character Types</label>
                        <div class="checkbox-group">
                            <label class="checkbox-label">
                                <input type="checkbox" id="uppercase" checked> Uppercase Letters (A-Z)
                            </label>
                            <label class="checkbox-label">
                                <input type="checkbox" id="lowercase" checked> Lowercase Letters (a-z)
                            </label>
                            <label class="checkbox-label">
                                <input type="checkbox" id="numbers" checked> Numbers (0-9)
                            </label>
                            <label class="checkbox-label">
                                <input type="checkbox" id="symbols" checked> Symbols (!@#$%^&*)
                            </label>
                        </div>
                    </div>
                    
                    <div class="input-group">
                        <label for="excludeChars">Exclude Characters</label>
                        <input type="text" id="excludeChars" placeholder="Characters to exclude (optional)" maxlength="20">
                        <small>Common exclusions: 0, O, 1, l, I</small>
                    </div>
                    
                    <div class="input-group">
                        <label for="passwordCount">Number of Passwords</label>
                        <input type="number" id="passwordCount" min="1" max="20" value="1">
                    </div>
                    
                    <button onclick="generatePasswords()" class="btn btn-primary">Generate Passwords</button>
                    
                    <div class="result" id="passwordResult" style="display: none;">
                        <h3>Generated Passwords</h3>
                        <div id="passwordList"></div>
                        <div class="password-actions">
                            <button onclick="copyAllPasswords()" class="btn btn-secondary">Copy All</button>
                            <button onclick="regeneratePasswords()" class="btn btn-secondary">Regenerate</button>
                        </div>
                        <div id="strengthIndicator"></div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Content Sections -->
    <section class="content-section">
        <div class="container">
            <div class="content-grid">
                <article class="content-card">
                    <h2>What is a Password Generator?</h2>
                    <p>A password generator is a specialized tool that creates strong, random passwords using advanced algorithms to ensure maximum security. Unlike manually created passwords that often follow predictable patterns, generated passwords are cryptographically secure and virtually impossible to guess or crack through brute force attacks.</p>
                    
                    <p>Modern password generators create passwords that combine multiple character types including uppercase letters, lowercase letters, numbers, and special symbols. This complexity exponentially increases the time and computational power required to crack the password, making your accounts significantly more secure.</p>
                    
                    <p>Our Password Generator is specifically designed for GoHighLevel users who need to:</p>
                    <ul>
                        <li><strong>Secure Client Accounts:</strong> Create strong passwords for client GoHighLevel instances</li>
                        <li><strong>Protect Agency Access:</strong> Generate secure passwords for team member accounts</li>
                        <li><strong>Maintain Compliance:</strong> Meet security requirements for data protection regulations</li>
                        <li><strong>Prevent Breaches:</strong> Reduce the risk of unauthorized access to sensitive client data</li>
                        <li><strong>Streamline Security:</strong> Quickly generate multiple secure passwords for different purposes</li>
                        <li><strong>Educate Clients:</strong> Demonstrate security best practices to clients</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Why GoHighLevel Users Need Strong Passwords</h2>
                    <p>As a GoHighLevel user or agency, you're managing sensitive client data, financial information, and business communications. Strong password security is absolutely critical for several reasons:</p>
                    
                    <h3>Client Data Protection</h3>
                    <p>Your GoHighLevel instances contain valuable client information including contact details, sales data, marketing campaigns, and financial records. A weak password could expose this sensitive information to unauthorized access, potentially resulting in data breaches and legal liabilities.</p>
                    
                    <h3>Agency Security Requirements</h3>
                    <p>Marketing agencies often handle multiple client accounts, making them attractive targets for cybercriminals. Strong passwords help protect your agency's reputation and maintain client trust by ensuring their data remains secure.</p>
                    
                    <h3>Compliance and Regulations</h3>
                    <p>Many industries have specific security requirements for handling client data. Strong passwords help meet these compliance standards and demonstrate your commitment to data security.</p>
                    
                    <h3>Business Continuity</h3>
                    <p>Account compromises can disrupt your business operations, damage client relationships, and result in significant financial losses. Strong passwords are your first line of defense against these threats.</p>
                    
                    <p>By using our Password Generator, you can create passwords that meet or exceed industry security standards, protecting both your business and your clients' valuable information.</p>
                </article>

                <article class="content-card">
                    <h2>How to Use the Password Generator</h2>
                    <p>Our Password Generator is designed to be both powerful and user-friendly. Here's how to create the most secure passwords for your needs:</p>
                    
                    <ol>
                        <li><strong>Set Password Length:</strong> Choose between 8-64 characters (16+ recommended for maximum security)</li>
                        <li><strong>Select Character Types:</strong> Choose which character sets to include in your passwords</li>
                        <li><strong>Exclude Problematic Characters:</strong> Remove characters that might cause confusion (like 0/O, 1/l/I)</li>
                        <li><strong>Choose Quantity:</strong> Generate multiple passwords for different accounts or purposes</li>
                        <li><strong>Generate and Review:</strong> Create passwords and assess their strength</li>
                        <li><strong>Copy and Store:</strong> Securely copy passwords to your password manager or secure storage</li>
                        <li><strong>Regenerate if Needed:</strong> Create new variations if you need different characteristics</li>
                    </ol>
                    
                    <h3>Pro Tips for Maximum Security</h3>
                    <ul>
                        <li><strong>Length Matters:</strong> Longer passwords are exponentially more secure than shorter ones</li>
                        <li><strong>Character Variety:</strong> Include all character types for maximum complexity</li>
                        <li><strong>Avoid Patterns:</strong> Don't use predictable sequences or common words</li>
                        <li><strong>Unique Per Account:</strong> Never reuse passwords across multiple accounts</li>
                        <li><strong>Regular Updates:</strong> Change passwords periodically, especially for critical accounts</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Use Cases for GoHighLevel Agencies</h2>
                    <p>Password generation is essential for GoHighLevel agencies managing multiple client accounts and team members:</p>
                    
                    <div class="use-cases">
                        <h3>Client Account Setup</h3>
                        <p>Generate secure passwords when setting up new GoHighLevel instances for clients, ensuring each account has a unique, strong password that meets security standards.</p>
                        
                        <h3>Team Member Access</h3>
                        <p>Create secure passwords for team members who need access to client accounts, maintaining security while enabling necessary collaboration.</p>
                        
                        <h3>Client Onboarding</h3>
                        <p>Provide clients with secure passwords for their new GoHighLevel accounts, demonstrating your commitment to security from day one.</p>
                        
                        <h3>Security Audits</h3>
                        <p>Generate new passwords during security reviews or when updating access protocols, maintaining the highest security standards.</p>
                        
                        <h3>Emergency Access</h3>
                        <p>Quickly generate secure passwords for emergency access situations or when resetting compromised accounts.</p>
                        
                        <h3>Client Education</h3>
                        <p>Use the generator to show clients how to create strong passwords, improving their overall security posture.</p>
                    </div>
                </article>

                <article class="content-card">
                    <h2>Password Security Best Practices</h2>
                    <p>Creating strong passwords is just the first step. Here are comprehensive best practices to maximize your security:</p>
                    
                    <h3>Password Creation Standards</h3>
                    <ul>
                        <li><strong>Minimum Length:</strong> Use at least 16 characters for maximum security</li>
                        <li><strong>Character Variety:</strong> Include uppercase, lowercase, numbers, and symbols</li>
                        <li><strong>Randomness:</strong> Avoid predictable patterns, sequences, or common words</li>
                        <li><strong>Uniqueness:</strong> Never reuse passwords across multiple accounts</li>
                        <li><strong>Complexity:</strong> Avoid personal information like names, birthdays, or addresses</li>
                    </ul>
                    
                    <h3>Password Management</h3>
                    <ul>
                        <li><strong>Use a Password Manager:</strong> Store passwords securely in a reputable password manager</li>
                        <li><strong>Enable Two-Factor Authentication:</strong> Add an extra layer of security beyond passwords</li>
                        <li><strong>Regular Updates:</strong> Change passwords every 90-180 days for critical accounts</li>
                        <li><strong>Secure Sharing:</strong> Share passwords securely with team members when necessary</li>
                        <li><strong>Monitor for Breaches:</strong> Use breach notification services to stay informed</li>
                    </ul>
                    
                    <h3>Common Password Mistakes to Avoid</h3>
                    <ul>
                        <li>Using easily guessable information (names, dates, common words)</li>
                        <li>Creating patterns like "123456" or "qwerty"</li>
                        <li>Using the same password for multiple accounts</li>
                        <li>Sharing passwords through insecure channels</li>
                        <li>Writing passwords down in plain text</li>
                        <li>Using passwords that are too short or simple</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Integration with GoHighLevel Security</h2>
                    <p>Our Password Generator integrates seamlessly with GoHighLevel's security features to create a comprehensive security strategy:</p>
                    
                    <h3>Account Security Enhancement</h3>
                    <p>Use generated passwords to strengthen your GoHighLevel account security:</p>
                    <ul>
                        <li>Create strong master passwords for your main GoHighLevel account</li>
                        <li>Generate secure passwords for client sub-accounts</li>
                        <li>Set up strong passwords for API access and integrations</li>
                        <li>Secure access to backup and export functions</li>
                    </ul>
                    
                    <h3>Team Security Management</h3>
                    <p>Maintain security across your team and client accounts:</p>
                    <ul>
                        <li>Generate unique passwords for each team member</li>
                        <li>Create secure passwords for client portal access</li>
                        <li>Set up strong passwords for shared resources</li>
                        <li>Maintain secure access to client data and campaigns</li>
                    </ul>
                    
                    <h3>Compliance and Auditing</h3>
                    <p>Meet security requirements and maintain audit trails:</p>
                    <ul>
                        <li>Document password policies and procedures</li>
                        <li>Maintain records of password changes and updates</li>
                        <li>Demonstrate security compliance to clients and regulators</li>
                        <li>Track access to sensitive client information</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Advanced Password Security Strategies</h2>
                    <p>Beyond basic password generation, consider these advanced strategies to maximize your security posture:</p>
                    
                    <h3>Password Strength Analysis</h3>
                    <p>Understand what makes a password truly secure:</p>
                    <ul>
                        <li><strong>Entropy:</strong> Measure the randomness and unpredictability of your passwords</li>
                        <li><strong>Pattern Recognition:</strong> Avoid common patterns that attackers can exploit</li>
                        <li><strong>Dictionary Attacks:</strong> Ensure passwords can't be found in common word lists</li>
                        <li><strong>Brute Force Resistance:</strong> Calculate how long it would take to crack your password</li>
                    </ul>
                    
                    <h3>Multi-Layer Security</h3>
                    <p>Combine strong passwords with additional security measures:</p>
                    <ul>
                        <li>Enable two-factor authentication on all accounts</li>
                        <li>Use hardware security keys for critical accounts</li>
                        <li>Implement IP restrictions and access controls</li>
                        <li>Monitor account activity for suspicious behavior</li>
                        <li>Regular security audits and penetration testing</li>
                    </ul>
                    
                    <h3>Password Recovery Strategies</h3>
                    <p>Plan for secure password recovery and account access:</p>
                    <ul>
                        <li>Set up secure recovery email addresses</li>
                        <li>Use security questions that can't be easily researched</li>
                        <li>Implement account recovery codes and backup access</li>
                        <li>Have emergency access procedures for critical accounts</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Industry Standards and Security Benchmarks</h2>
                    <p>Understanding industry standards helps you set appropriate security goals and measure your security posture:</p>
                    
                    <h3>Password Strength Standards</h3>
                    <ul>
                        <li><strong>Minimum Length:</strong> 12-16 characters for most accounts, 20+ for critical systems</li>
                        <li><strong>Character Requirements:</strong> Mix of uppercase, lowercase, numbers, and symbols</li>
                        <li><strong>Complexity Score:</strong> Aim for 80+ on password strength meters</li>
                        <li><strong>Uniqueness:</strong> 100% unique passwords across all accounts</li>
                        <li><strong>Update Frequency:</strong> Every 90-180 days for critical accounts</li>
                    </ul>
                    
                    <h3>Security Breach Statistics</h3>
                    <ul>
                        <li><strong>Weak Passwords:</strong> Account for 81% of data breaches</li>
                        <li><strong>Password Reuse:</strong> 65% of people reuse passwords across accounts</li>
                        <li><strong>Brute Force Attacks:</strong> 23% of attacks use automated password guessing</li>
                        <li><strong>Social Engineering:</strong> 14% of breaches involve password manipulation</li>
                        <li><strong>Average Time to Crack:</strong> Weak passwords can be cracked in seconds</li>
                    </ul>
                    
                    <h3>Security Investment ROI</h3>
                    <ul>
                        <li><strong>Prevention Cost:</strong> $1 spent on prevention saves $10 in incident response</li>
                        <li><strong>Data Breach Cost:</strong> Average cost of $3.86 million per breach</li>
                        <li><strong>Customer Trust:</strong> 85% of customers won't do business with breached companies</li>
                        <li><strong>Compliance Fines:</strong> GDPR fines can reach €20 million or 4% of global revenue</li>
                        <li><strong>Recovery Time:</strong> Average 280 days to identify and contain a breach</li>
                    </ul>
                    
                    <p><strong>Remember:</strong> The cost of implementing strong password security is minimal compared to the potential cost of a security breach. Investing in password security is investing in your business's future.</p>
                </article>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="related-tools">
        <div class="container">
            <h2>Related Tools</h2>
            <div class="tools-grid">
                <a href="email-subject-line-scorer.php" class="tool-link">
                    <h3>Email Subject Line Scorer</h3>
                    <p>Optimize your email campaigns for better security and engagement</p>
                </a>
                <a href="lead-scoring.php" class="tool-link">
                    <h3>Lead Scoring Tool</h3>
                    <p>Secure your lead data with proper access controls and strong passwords</p>
                </a>
                <a href="utm-builder.php" class="tool-link">
                    <h3>UTM Builder</h3>
                    <p>Track campaign performance while maintaining security standards</p>
                </a>
                <a href="roi-calculator.php" class="tool-link">
                    <h3>ROI Calculator</h3>
                    <p>Calculate the ROI of your security investments and password policies</p>
                </a>
            </div>
        </div>
    </section>

    <?php inc('includes/footer.php'); ?>

    <script src="common.js"></script>
    <script>
        // Update length display
        document.getElementById('passwordLength').addEventListener('input', function() {
            document.getElementById('lengthValue').textContent = this.value;
        });

        function generatePasswords() {
            const length = parseInt(document.getElementById('passwordLength').value);
            const useUppercase = document.getElementById('uppercase').checked;
            const useLowercase = document.getElementById('lowercase').checked;
            const useNumbers = document.getElementById('numbers').checked;
            const useSymbols = document.getElementById('symbols').checked;
            const excludeChars = document.getElementById('excludeChars').value;
            const count = parseInt(document.getElementById('passwordCount').value);
            
            if (!useUppercase && !useLowercase && !useNumbers && !useSymbols) {
                alert('Please select at least one character type.');
                return;
            }
            
            const passwords = [];
            for (let i = 0; i < count; i++) {
                passwords.push(generatePassword(length, useUppercase, useLowercase, useNumbers, useSymbols, excludeChars));
            }
            
            displayPasswords(passwords);
            analyzePasswordStrength(passwords[0]);
            document.getElementById('passwordResult').style.display = 'block';
        }
        
        function generatePassword(length, useUppercase, useLowercase, useNumbers, useSymbols, excludeChars) {
            let chars = '';
            if (useUppercase) chars += 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
            if (useLowercase) chars += 'abcdefghijklmnopqrstuvwxyz';
            if (useNumbers) chars += '0123456789';
            if (useSymbols) chars += '!@#$%^&*()_+-=[]{}|;:,.<>?';
            
            // Remove excluded characters
            if (excludeChars) {
                for (let char of excludeChars) {
                    chars = chars.replace(new RegExp(char, 'g'), '');
                }
            }
            
            let password = '';
            for (let i = 0; i < length; i++) {
                password += chars.charAt(Math.floor(Math.random() * chars.length));
            }
            
            return password;
        }
        
        function displayPasswords(passwords) {
            const passwordList = document.getElementById('passwordList');
            passwordList.innerHTML = '';
            
            passwords.forEach((password, index) => {
                const passwordDiv = document.createElement('div');
                passwordDiv.className = 'password-item';
                passwordDiv.innerHTML = `
                    <span class="password-text">${password}</span>
                    <button onclick="copyPassword('${password}')" class="btn btn-small">Copy</button>
                `;
                passwordList.appendChild(passwordDiv);
            });
        }
        
        function copyPassword(password) {
            navigator.clipboard.writeText(password).then(() => {
                alert('Password copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy password: ', err);
                // Fallback for older browsers
                const textArea = document.createElement('textarea');
                textArea.value = password;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('Password copied to clipboard!');
            });
        }
        
        function copyAllPasswords() {
            const passwordElements = document.querySelectorAll('.password-text');
            const allPasswords = Array.from(passwordElements).map(el => el.textContent).join('\n');
            
            navigator.clipboard.writeText(allPasswords).then(() => {
                alert('All passwords copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy passwords: ', err);
                alert('Failed to copy passwords. Please copy them individually.');
            });
        }
        
        function regeneratePasswords() {
            generatePasswords();
        }
        
        function analyzePasswordStrength(password) {
            let score = 0;
            let feedback = [];
            
            // Length scoring
            if (password.length >= 16) {
                score += 25;
                feedback.push('Excellent length');
            } else if (password.length >= 12) {
                score += 20;
                feedback.push('Good length');
            } else if (password.length >= 8) {
                score += 15;
                feedback.push('Acceptable length');
            } else {
                score += 5;
                feedback.push('Too short');
            }
            
            // Character variety scoring
            let hasUpper = /[A-Z]/.test(password);
            let hasLower = /[a-z]/.test(password);
            let hasNumber = /[0-9]/.test(password);
            let hasSymbol = /[^A-Za-z0-9]/.test(password);
            
            let varietyCount = [hasUpper, hasLower, hasNumber, hasSymbol].filter(Boolean).length;
            score += varietyCount * 5;
            
            if (varietyCount === 4) feedback.push('All character types included');
            else if (varietyCount === 3) feedback.push('Good character variety');
            else if (varietyCount === 2) feedback.push('Limited character variety');
            else feedback.push('Very limited character variety');
            
            // Randomness scoring
            let consecutiveCount = 0;
            for (let i = 1; i < password.length; i++) {
                if (password.charCodeAt(i) === password.charCodeAt(i-1) + 1) {
                    consecutiveCount++;
                }
            }
            
            if (consecutiveCount === 0) {
                score += 20;
                feedback.push('No consecutive characters');
            } else if (consecutiveCount <= 2) {
                score += 15;
                feedback.push('Minimal consecutive characters');
            } else {
                score += 5;
                feedback.push('Multiple consecutive characters');
            }
            
            // Overall assessment
            let strength = '';
            let color = '';
            
            if (score >= 80) {
                strength = 'Very Strong';
                color = '#27ae60';
            } else if (score >= 60) {
                strength = 'Strong';
                color = '#2ecc71';
            } else if (score >= 40) {
                strength = 'Moderate';
                color = '#f39c12';
            } else if (score >= 20) {
                strength = 'Weak';
                color = '#e67e22';
            } else {
                strength = 'Very Weak';
                color = '#e74c3c';
            }
            
            const strengthIndicator = document.getElementById('strengthIndicator');
            strengthIndicator.innerHTML = `
                <h4>Password Strength Analysis</h4>
                <div class="strength-display">
                    <p class="strength-score" style="color: ${color}">${strength} (${score}/100)</p>
                    <div class="strength-bar">
                        <div class="strength-fill" style="width: ${score}%; background-color: ${color}"></div>
                    </div>
                    <ul class="strength-feedback">
                        ${feedback.map(f => `<li>${f}</li>`).join('')}
                    </ul>
                </div>
            `;
        }
    </script>
</body>
</html>

