/**
 * Consent Management System for GHL Toolset
 * Handles GDPR-compliant cookie consent with granular controls
 */

(() => {
  'use strict';

  const CONSENT_VERSION = '1.0';
  const STORAGE_KEY = `consent.v${CONSENT_VERSION}`;
  const CONSENT_EXPIRY_DAYS = 365;

  // Default consent state
  const defaultConsent = {
    necessary: true,    // Always true, cannot be disabled
    analytics: false,
    ads: false,
    preferences: false,
    marketing: false,
    version: CONSENT_VERSION,
    timestamp: null
  };

  // Current consent state
  let consentState = { ...defaultConsent };

  // Global consent object for other scripts
  window.__consent = consentState;

  // Script loading flags
  window.__scriptsLoaded = {
    ga4: false,
    adsense: false
  };

  /**
   * Initialize consent system
   */
  function init() {
    loadConsentState();
    setupConsentBanner();
    setupConsentModal();
    
    // Auto-load scripts if consent already given
    if (consentState.analytics && !window.__scriptsLoaded.ga4) {
      loadGA4();
    }
    if (consentState.ads && !window.__scriptsLoaded.adsense) {
      loadAdsense();
    }

    // Expose methods globally
    window.setConsent = setConsent;
    window.getConsent = getConsent;
    window.showConsentModal = showConsentModal;
    window.revokeConsent = revokeConsent;

    // Dispatch consent ready event
    window.dispatchEvent(new CustomEvent('consentReady', { detail: consentState }));
  }

  /**
   * Load consent state from localStorage
   */
  function loadConsentState() {
    try {
      const stored = localStorage.getItem(STORAGE_KEY);
      if (stored) {
        const parsed = JSON.parse(stored);
        
        // Check if consent is still valid
        if (parsed.version === CONSENT_VERSION && 
            parsed.timestamp && 
            Date.now() - parsed.timestamp < CONSENT_EXPIRY_DAYS * 24 * 60 * 60 * 1000) {
          
          consentState = { ...defaultConsent, ...parsed };
          window.__consent = consentState;
          return;
        }
      }
    } catch (error) {
      console.warn('Failed to load consent state:', error);
    }

    // Show banner if no valid consent
    if (!consentState.timestamp) {
      showConsentBanner();
    }
  }

  /**
   * Save consent state to localStorage
   */
  function saveConsentState() {
    try {
      consentState.timestamp = Date.now();
      localStorage.setItem(STORAGE_KEY, JSON.stringify(consentState));
      
      // Also set a cookie for server-side access
      const cookieValue = encodeURIComponent(JSON.stringify({
        analytics: consentState.analytics,
        ads: consentState.ads,
        timestamp: consentState.timestamp
      }));
      
      document.cookie = `consent=${cookieValue}; max-age=${CONSENT_EXPIRY_DAYS * 24 * 60 * 60}; path=/; secure; samesite=strict`;
      
    } catch (error) {
      console.error('Failed to save consent state:', error);
    }
  }

  /**
   * Update consent preferences
   */
  function setConsent(updates) {
    const prevState = { ...consentState };
    
    Object.assign(consentState, updates);
    consentState.necessary = true; // Always true
    
    window.__consent = consentState;
    saveConsentState();

    // Load scripts based on new consent
    if (consentState.analytics && !prevState.analytics && !window.__scriptsLoaded.ga4) {
      loadGA4();
    }
    
    if (consentState.ads && !prevState.ads && !window.__scriptsLoaded.adsense) {
      loadAdsense();
    }

    // Hide banner/modal
    hideConsentBanner();
    hideConsentModal();

    // Dispatch consent change event
    window.dispatchEvent(new CustomEvent('consentChanged', { 
      detail: { 
        previous: prevState, 
        current: consentState,
        changes: updates
      } 
    }));

    // Track consent acceptance
    if (window.track && (updates.analytics || updates.ads)) {
      setTimeout(() => {
        window.track('consent_given', {
          analytics: consentState.analytics,
          ads: consentState.ads,
          marketing: consentState.marketing,
          preferences: consentState.preferences
        });
      }, 100);
    }
  }

  /**
   * Get current consent state
   */
  function getConsent() {
    return { ...consentState };
  }

  /**
   * Revoke all consent
   */
  function revokeConsent() {
    setConsent({
      analytics: false,
      ads: false,
      marketing: false,
      preferences: false
    });

    // Clear analytics cookies
    if (window.gtag) {
      window.gtag('consent', 'update', {
        analytics_storage: 'denied',
        ad_storage: 'denied'
      });
    }

    showConsentBanner();
  }

  /**
   * Load Google Analytics 4
   */
  function loadGA4() {
    if (window.__scriptsLoaded.ga4 || !window.GA4_ID) return;
    
    window.__scriptsLoaded.ga4 = true;
    
    // Initialize dataLayer
    window.dataLayer = window.dataLayer || [];
    function gtag() { dataLayer.push(arguments); }
    window.gtag = gtag;

    // Configure consent mode
    gtag('consent', 'default', {
      analytics_storage: consentState.analytics ? 'granted' : 'denied',
      ad_storage: consentState.ads ? 'granted' : 'denied',
      functionality_storage: consentState.preferences ? 'granted' : 'denied',
      personalization_storage: consentState.marketing ? 'granted' : 'denied',
      security_storage: 'granted'
    });

    // Initialize GA4
    gtag('js', new Date());
    gtag('config', window.GA4_ID, {
      send_page_view: true,
      cookie_flags: 'secure;samesite=strict',
      anonymize_ip: true
    });

    // Load GA4 script
    loadScript(`https://www.googletagmanager.com/gtag/js?id=${window.GA4_ID}`, () => {
      console.log('GA4 loaded successfully');
    });
  }

  /**
   * Load Google AdSense
   */
  function loadAdsense() {
    if (window.__scriptsLoaded.adsense || !window.ADSENSE_PUB) return;
    
    window.__scriptsLoaded.adsense = true;

    // Load AdSense script
    loadScript(`https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js?client=${window.ADSENSE_PUB}`, () => {
      console.log('AdSense loaded successfully');
      
      // Initialize existing ad slots
      const adSlots = document.querySelectorAll('ins.adsbygoogle:not([data-adsbygoogle-status])');
      adSlots.forEach(() => {
        try {
          (window.adsbygoogle = window.adsbygoogle || []).push({});
        } catch (error) {
          console.warn('Failed to initialize ad slot:', error);
        }
      });

      // Show ad slots and hide placeholders
      document.querySelectorAll('.ad-slot').forEach(slot => {
        const placeholder = slot.querySelector('.ad-placeholder');
        if (placeholder) {
          placeholder.style.display = 'none';
        }
      });
    });
  }

  /**
   * Load external script with error handling
   */
  function loadScript(src, callback) {
    const script = document.createElement('script');
    script.src = src;
    script.async = true;
    script.onload = callback;
    script.onerror = () => console.error(`Failed to load script: ${src}`);
    document.head.appendChild(script);
  }

  /**
   * Create and show consent banner
   */
  function setupConsentBanner() {
    if (document.getElementById('consent-banner')) return;

    const banner = document.createElement('div');
    banner.id = 'consent-banner';
    banner.className = 'consent-banner';
    banner.setAttribute('role', 'banner');
    banner.setAttribute('aria-label', 'Cookie consent');
    
    banner.innerHTML = `
      <div class="consent-banner-content">
        <div class="consent-banner-text">
          <h3>We value your privacy</h3>
          <p>We use cookies to enhance your experience, analyze site traffic, and for marketing purposes. 
             You can customize your preferences or accept all cookies.</p>
        </div>
        <div class="consent-banner-actions">
          <button type="button" class="btn btn-secondary" data-action="customize">
            Customize
          </button>
          <button type="button" class="btn btn-primary" data-action="accept-all">
            Accept All
          </button>
          <button type="button" class="btn btn-outline" data-action="reject-all">
            Reject All
          </button>
        </div>
      </div>
    `;

    // Add event listeners
    banner.querySelector('[data-action="accept-all"]').addEventListener('click', () => {
      setConsent({
        analytics: true,
        ads: true,
        marketing: true,
        preferences: true
      });
    });

    banner.querySelector('[data-action="reject-all"]').addEventListener('click', () => {
      setConsent({
        analytics: false,
        ads: false,
        marketing: false,
        preferences: false
      });
    });

    banner.querySelector('[data-action="customize"]').addEventListener('click', showConsentModal);

    document.body.appendChild(banner);
  }

  /**
   * Show consent banner
   */
  function showConsentBanner() {
    const banner = document.getElementById('consent-banner');
    if (banner) {
      banner.style.display = 'block';
      banner.setAttribute('aria-hidden', 'false');
    } else {
      setupConsentBanner();
    }
  }

  /**
   * Hide consent banner
   */
  function hideConsentBanner() {
    const banner = document.getElementById('consent-banner');
    if (banner) {
      banner.style.display = 'none';
      banner.setAttribute('aria-hidden', 'true');
    }
  }

  /**
   * Create consent preferences modal
   */
  function setupConsentModal() {
    if (document.getElementById('consent-modal')) return;

    const modal = document.createElement('div');
    modal.id = 'consent-modal';
    modal.className = 'consent-modal';
    modal.setAttribute('role', 'dialog');
    modal.setAttribute('aria-modal', 'true');
    modal.setAttribute('aria-labelledby', 'consent-modal-title');
    modal.style.display = 'none';

    modal.innerHTML = `
      <div class="consent-modal-backdrop"></div>
      <div class="consent-modal-content">
        <div class="consent-modal-header">
          <h2 id="consent-modal-title">Privacy Preferences</h2>
          <button type="button" class="consent-modal-close" aria-label="Close">×</button>
        </div>
        <div class="consent-modal-body">
          <p>Manage your cookie preferences. You can change these settings at any time.</p>
          
          <div class="consent-option">
            <div class="consent-option-header">
              <label>
                <input type="checkbox" checked disabled>
                <span class="consent-option-title">Necessary Cookies</span>
              </label>
            </div>
            <p class="consent-option-description">
              Essential for the website to function properly. Cannot be disabled.
            </p>
          </div>

          <div class="consent-option">
            <div class="consent-option-header">
              <label>
                <input type="checkbox" id="consent-analytics">
                <span class="consent-option-title">Analytics Cookies</span>
              </label>
            </div>
            <p class="consent-option-description">
              Help us understand how visitors interact with our website by collecting anonymous information.
            </p>
          </div>

          <div class="consent-option">
            <div class="consent-option-header">
              <label>
                <input type="checkbox" id="consent-ads">
                <span class="consent-option-title">Advertisement Cookies</span>
              </label>
            </div>
            <p class="consent-option-description">
              Used to deliver relevant advertisements and support our free tools.
            </p>
          </div>

          <div class="consent-option">
            <div class="consent-option-header">
              <label>
                <input type="checkbox" id="consent-marketing">
                <span class="consent-option-title">Marketing Cookies</span>
              </label>
            </div>
            <p class="consent-option-description">
              Used to track visitors across websites to display relevant marketing content.
            </p>
          </div>

          <div class="consent-option">
            <div class="consent-option-header">
              <label>
                <input type="checkbox" id="consent-preferences">
                <span class="consent-option-title">Preference Cookies</span>
              </label>
            </div>
            <p class="consent-option-description">
              Remember your settings and preferences for a better experience.
            </p>
          </div>
        </div>
        <div class="consent-modal-footer">
          <button type="button" class="btn btn-outline" data-action="save-preferences">
            Save Preferences
          </button>
          <button type="button" class="btn btn-primary" data-action="accept-all-modal">
            Accept All
          </button>
        </div>
      </div>
    `;

    // Add event listeners
    const closeBtn = modal.querySelector('.consent-modal-close');
    const backdrop = modal.querySelector('.consent-modal-backdrop');
    const saveBtn = modal.querySelector('[data-action="save-preferences"]');
    const acceptAllBtn = modal.querySelector('[data-action="accept-all-modal"]');

    [closeBtn, backdrop].forEach(el => {
      el.addEventListener('click', hideConsentModal);
    });

    saveBtn.addEventListener('click', () => {
      const preferences = {
        analytics: modal.querySelector('#consent-analytics').checked,
        ads: modal.querySelector('#consent-ads').checked,
        marketing: modal.querySelector('#consent-marketing').checked,
        preferences: modal.querySelector('#consent-preferences').checked
      };
      setConsent(preferences);
    });

    acceptAllBtn.addEventListener('click', () => {
      setConsent({
        analytics: true,
        ads: true,
        marketing: true,
        preferences: true
      });
    });

    // Keyboard navigation
    modal.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') {
        hideConsentModal();
      }
    });

    document.body.appendChild(modal);
  }

  /**
   * Show consent modal
   */
  function showConsentModal() {
    const modal = document.getElementById('consent-modal');
    if (!modal) {
      setupConsentModal();
      return showConsentModal();
    }

    // Update checkboxes with current state
    modal.querySelector('#consent-analytics').checked = consentState.analytics;
    modal.querySelector('#consent-ads').checked = consentState.ads;
    modal.querySelector('#consent-marketing').checked = consentState.marketing;
    modal.querySelector('#consent-preferences').checked = consentState.preferences;

    modal.style.display = 'block';
    modal.setAttribute('aria-hidden', 'false');
    
    // Focus management
    const firstFocusable = modal.querySelector('button, input');
    if (firstFocusable) {
      firstFocusable.focus();
    }

    // Prevent body scroll
    document.body.style.overflow = 'hidden';
  }

  /**
   * Hide consent modal
   */
  function hideConsentModal() {
    const modal = document.getElementById('consent-modal');
    if (modal) {
      modal.style.display = 'none';
      modal.setAttribute('aria-hidden', 'true');
      document.body.style.overflow = '';
    }
  }

  // Initialize when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Add consent banner styles
  const styles = document.createElement('style');
  styles.textContent = `
    .consent-banner {
      position: fixed;
      bottom: 0;
      left: 0;
      right: 0;
      background: white;
      border-top: 1px solid var(--color-neutral-200);
      box-shadow: var(--shadow-lg);
      z-index: var(--z-fixed);
      padding: var(--space-4);
    }

    .consent-banner-content {
      max-width: var(--container-xl);
      margin: 0 auto;
      display: flex;
      align-items: center;
      gap: var(--space-6);
    }

    .consent-banner-text {
      flex: 1;
    }

    .consent-banner-text h3 {
      margin: 0 0 var(--space-2) 0;
      font-size: var(--font-size-lg);
      font-weight: var(--font-weight-semibold);
    }

    .consent-banner-text p {
      margin: 0;
      color: var(--color-neutral-600);
      font-size: var(--font-size-sm);
    }

    .consent-banner-actions {
      display: flex;
      gap: var(--space-3);
      flex-shrink: 0;
    }

    .consent-modal {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      z-index: var(--z-modal);
    }

    .consent-modal-backdrop {
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.5);
      backdrop-filter: blur(4px);
    }

    .consent-modal-content {
      position: relative;
      max-width: 600px;
      margin: 10vh auto;
      background: white;
      border-radius: var(--radius-xl);
      box-shadow: var(--shadow-2xl);
      max-height: 80vh;
      overflow: hidden;
      display: flex;
      flex-direction: column;
    }

    .consent-modal-header {
      padding: var(--space-6);
      border-bottom: 1px solid var(--color-neutral-200);
      display: flex;
      align-items: center;
      justify-content: space-between;
    }

    .consent-modal-header h2 {
      margin: 0;
      font-size: var(--font-size-xl);
    }

    .consent-modal-close {
      background: none;
      border: none;
      font-size: var(--font-size-2xl);
      cursor: pointer;
      padding: var(--space-2);
      line-height: 1;
    }

    .consent-modal-body {
      padding: var(--space-6);
      overflow-y: auto;
      flex: 1;
    }

    .consent-option {
      margin-bottom: var(--space-6);
    }

    .consent-option-header {
      margin-bottom: var(--space-2);
    }

    .consent-option-header label {
      display: flex;
      align-items: center;
      gap: var(--space-3);
      cursor: pointer;
      font-weight: var(--font-weight-medium);
    }

    .consent-option-description {
      margin: 0;
      font-size: var(--font-size-sm);
      color: var(--color-neutral-600);
      margin-left: calc(var(--space-3) + 1rem);
    }

    .consent-modal-footer {
      padding: var(--space-6);
      border-top: 1px solid var(--color-neutral-200);
      display: flex;
      gap: var(--space-3);
      justify-content: flex-end;
    }

    @media (max-width: 768px) {
      .consent-banner-content {
        flex-direction: column;
        text-align: center;
        gap: var(--space-4);
      }

      .consent-banner-actions {
        justify-content: center;
        flex-wrap: wrap;
      }

      .consent-modal-content {
        margin: var(--space-4);
        max-height: calc(100vh - var(--space-8));
      }
    }
  `;
  
  document.head.appendChild(styles);

})();