﻿<?php
require_once __DIR__ . '/config.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>JSON Formatter & Validator for GoHighLevel | Free JSON Tool</title>
    <meta name="description" content="Free JSON formatter and validator for GoHighLevel users. Format, validate, and beautify JSON data. Perfect for agencies working with APIs and data integration.">
    <meta name="keywords" content="json formatter, json validator, json beautifier, json prettifier, GoHighLevel agency, API integration">
    <meta name="author" content="Paddle CRM">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph / Facebook -->
    <meta property="og:type" content="website">
    <meta property="og:title" content="JSON Formatter & Validator for GoHighLevel">
    <meta property="og:description" content="Free JSON formatter and validator for GoHighLevel users. Format, validate, and beautify JSON data.">
    <meta property="og:url" content="https://yourdomain.com/json-formatter-validator.php">
    <meta property="og:image" content="https://yourdomain.com/images/json-formatter-validator-og.jpg">
    
    <!-- Twitter -->
    <meta property="twitter:card" content="summary_large_image">
    <meta property="twitter:title" content="JSON Formatter & Validator for GoHighLevel">
    <meta property="twitter:description" content="Free JSON formatter and validator for GoHighLevel users. Format, validate, and beautify JSON data.">
    <meta property="twitter:image" content="https://yourdomain.com/images/json-formatter-validator-og.jpg">
    
    <!-- Schema.org structured data -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "JSON Formatter & Validator for GoHighLevel",
        "description": "Free tool to format, validate, and beautify JSON data",
        "url": "https://yourdomain.com/json-formatter-validator.php",
        "applicationCategory": "BusinessApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Paddle CRM",
            "description": "WhiteLabel GoHighLevel Agency"
        }
    }
    </script>
    
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <?php inc('includes/header.php'); ?>

    <!-- Hero Section -->
    <section class="hero tool-hero">
        <div class="container">
            <nav class="breadcrumb">
                <a href="index.php">Home</a> > <span>JSON Formatter & Validator</span>
            </nav>
            <div class="hero-content">
                <div class="hero-text">
                    <h1 class="hero-title">JSON Formatter & Validator for GoHighLevel</h1>
                    <p class="hero-subtitle">Format, validate, and beautify JSON data for better readability and API integration</p>
                    <div class="hero-stats">
                        <div class="stat-item">
                            <span class="stat-number">Professional</span>
                            <span class="stat-label">Quality</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">GoHighLevel</span>
                            <span class="stat-label">Optimized</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">100%</span>
                            <span class="stat-label">Free Tool</span>
                        </div>
                    </div>
                </div>
                <div class="hero-visual">
                    <div class="hero-image">
                        <div class="floating-card card-1">
                            <span class="card-icon">📊</span>
                            <span class="card-text">JSON</span>
                        </div>
                        <div class="floating-card card-2">
                            <span class="card-icon">⚡</span>
                            <span class="card-text">Fast</span>
                        </div>
                        <div class="floating-card card-3">
                            <span class="card-icon">🎯</span>
                            <span class="card-text">Validated</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Section -->
    <section class="tool-section">
        <div class="container">
            <div class="tool-card featured">
                <div class="tool-header">
                    <div class="tool-icon">📊</div>
                    <h2>JSON Formatter & Validator</h2>
                    <p>Professional JSON formatting and validation tool designed specifically for GoHighLevel users and agencies</p>
                </div>
                
                <div class="tool-interface">
                    <div class="json-tabs">
                        <button class="tab-btn active" onclick="showTab('format')">Format JSON</button>
                        <button class="tab-btn" onclick="showTab('validate')">Validate JSON</button>
                        <button class="tab-btn" onclick="showTab('minify')">Minify JSON</button>
                        <button class="tab-btn" onclick="showTab('convert')">Convert JSON</button>
                    </div>
                    
                    <div class="tab-content">
                        <!-- Format Tab -->
                        <div id="format-tab" class="tab-pane active">
                            <div class="input-group">
                                <label for="formatInput">JSON Input</label>
                                <textarea id="formatInput" rows="12" placeholder="Paste your JSON data here..."></textarea>
                                <div class="char-counter">
                                    <span id="formatCharCounter">0</span> characters
                                </div>
                            </div>
                            
                            <div class="input-group">
                                <label for="formatOptions">Formatting Options</label>
                                <div class="options-grid">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="formatIndent" checked>
                                        Indent Output
                                    </label>
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="formatSort" checked>
                                        Sort Keys
                                    </label>
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="formatCompact">
                                        Compact Arrays
                                    </label>
                                </div>
                            </div>
                            
                            <div class="button-group">
                                <button class="btn btn-primary" onclick="formatJSON()">Format JSON</button>
                                <button class="btn btn-secondary" onclick="clearFormat()">Clear</button>
                            </div>
                            
                            <div class="result" id="formatResult" style="display: none;">
                                <h3>Formatted JSON</h3>
                                <div class="result-display">
                                    <div class="result-item">
                                        <label>Formatted Output:</label>
                                        <div class="result-value result-output" id="formattedOutput">
                                            <pre id="formattedJSON"></pre>
                                            <button class="copy-btn" onclick="copyResult('formattedJSON')">📋 Copy</button>
                                        </div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Validation Status:</label>
                                        <div class="result-value" id="validationStatus"></div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Output Size:</label>
                                        <div class="result-value" id="outputSize"></div>
                                    </div>
                                </div>
                                
                                <div class="result-actions">
                                    <button class="btn btn-primary" onclick="downloadFormatted()">Download Result</button>
                                    <button class="btn btn-secondary" onclick="minifyFormatted()">Minify This</button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Validate Tab -->
                        <div id="validate-tab" class="tab-pane">
                            <div class="input-group">
                                <label for="validateInput">JSON to Validate</label>
                                <textarea id="validateInput" rows="12" placeholder="Paste JSON to validate..."></textarea>
                                <div class="char-counter">
                                    <span id="validateCharCounter">0</span> characters
                                </div>
                            </div>
                            
                            <div class="button-group">
                                <button class="btn btn-primary" onclick="validateJSON()">Validate JSON</button>
                                <button class="btn btn-secondary" onclick="clearValidate()">Clear</button>
                            </div>
                            
                            <div class="result" id="validateResult" style="display: none;">
                                <h3>Validation Results</h3>
                                <div class="result-display">
                                    <div class="result-item">
                                        <label>Validation Status:</label>
                                        <div class="result-value" id="validateStatus"></div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Error Details:</label>
                                        <div class="result-value" id="errorDetails"></div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>JSON Structure:</label>
                                        <div class="result-value" id="jsonStructure"></div>
                                    </div>
                                </div>
                                
                                <div class="result-actions">
                                    <button class="btn btn-primary" onclick="fixErrors()">Fix Errors</button>
                                    <button class="btn btn-secondary" onclick="formatValidated()">Format If Valid</button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Minify Tab -->
                        <div id="minify-tab" class="tab-pane">
                            <div class="input-group">
                                <label for="minifyInput">JSON to Minify</label>
                                <textarea id="minifyInput" rows="12" placeholder="Paste JSON to minify..."></textarea>
                                <div class="char-counter">
                                    <span id="minifyCharCounter">0</span> characters
                                </div>
                            </div>
                            
                            <div class="button-group">
                                <button class="btn btn-primary" onclick="minifyJSON()">Minify JSON</button>
                                <button class="btn btn-secondary" onclick="clearMinify()">Clear</button>
                            </div>
                            
                            <div class="result" id="minifyResult" style="display: none;">
                                <h3>Minified JSON</h3>
                                <div class="result-display">
                                    <div class="result-item">
                                        <label>Minified Output:</label>
                                        <div class="result-value result-output" id="minifiedOutput">
                                            <pre id="minifiedJSON"></pre>
                                            <button class="copy-btn" onclick="copyResult('minifiedJSON')">📋 Copy</button>
                                        </div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Size Reduction:</label>
                                        <div class="result-value" id="sizeReduction"></div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Compression Ratio:</label>
                                        <div class="result-value" id="compressionRatio"></div>
                                    </div>
                                </div>
                                
                                <div class="result-actions">
                                    <button class="btn btn-primary" onclick="downloadMinified()">Download Result</button>
                                    <button class="btn btn-secondary" onclick="formatMinified()">Format This</button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Convert Tab -->
                        <div id="convert-tab" class="tab-pane">
                            <div class="input-group">
                                <label for="convertInput">Data to Convert</label>
                                <textarea id="convertInput" rows="12" placeholder="Paste data to convert to JSON..."></textarea>
                                <div class="char-counter">
                                    <span id="convertCharCounter">0</span> characters
                                </div>
                            </div>
                            
                            <div class="input-group">
                                <label for="convertType">Input Type</label>
                                <select id="convertType">
                                    <option value="csv">CSV</option>
                                    <option value="xml">XML</option>
                                    <option value="yaml">YAML</option>
                                    <option value="query">Query String</option>
                                </select>
                            </div>
                            
                            <div class="button-group">
                                <button class="btn btn-primary" onclick="convertToJSON()">Convert to JSON</button>
                                <button class="btn btn-secondary" onclick="clearConvert()">Clear</button>
                            </div>
                            
                            <div class="result" id="convertResult" style="display: none;">
                                <h3>Converted JSON</h3>
                                <div class="result-display">
                                    <div class="result-item">
                                        <label>Converted Output:</label>
                                        <div class="result-value result-output" id="convertedOutput">
                                            <pre id="convertedJSON"></pre>
                                            <button class="copy-btn" onclick="copyResult('convertedJSON')">📋 Copy</button>
                                        </div>
                                    </div>
                                    
                                    <div class="result-item">
                                        <label>Conversion Status:</label>
                                        <div class="result-value" id="conversionStatus"></div>
                                    </div>
                                </div>
                                
                                <div class="result-actions">
                                    <button class="btn btn-primary" onclick="downloadConverted()">Download Result</button>
                                    <button class="btn btn-secondary" onclick="formatConverted()">Format This</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Content Section -->
    <section class="content-section">
        <div class="container">
            <div class="content-grid">
                <article class="content-card">
                    <h2>What is JSON?</h2>
                    <p>JSON (JavaScript Object Notation) is a lightweight, text-based data interchange format that is easy for humans to read and write, and easy for machines to parse and generate. It's based on a subset of JavaScript and is commonly used for transmitting data between a server and web application.</p>
                    
                    <p>For GoHighLevel users, understanding JSON is essential because:</p>
                    <ul>
                        <li><strong>API Integration:</strong> Most APIs use JSON for data exchange</li>
                        <li><strong>Data Handling:</strong> JSON is the standard format for web data</li>
                        <li><strong>System Integration:</strong> Connect different platforms and services</li>
                        <li><strong>Client Data:</strong> Handle various client data formats</li>
                        <li><strong>Web Development:</strong> Build modern web applications</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Why GoHighLevel Users Need This Tool</h2>
                    <p>As a GoHighLevel agency, you're constantly working with APIs, data integration, and client systems. JSON is the lingua franca of modern web development, and having tools to work with it efficiently is crucial for success.</p>
                    
                    <ul>
                        <li><strong>API Development:</strong> Create and consume APIs with proper JSON formatting</li>
                        <li><strong>Data Validation:</strong> Ensure data integrity in client communications</li>
                        <li><strong>System Integration:</strong> Connect GoHighLevel with other platforms</li>
                        <li><strong>Client Support:</strong> Help clients with JSON-related issues</li>
                        <li><strong>Data Migration:</strong> Convert data between different formats</li>
                        <li><strong>Testing and Debugging:</strong> Verify JSON structure in development</li>
                        <li><strong>Performance Optimization:</strong> Optimize JSON for better performance</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>How to Use the JSON Tool</h2>
                    <p>Our JSON tool provides comprehensive formatting, validation, and conversion capabilities:</p>
                    
                    <ol>
                        <li><strong>Choose Operation:</strong> Select between formatting, validation, minification, or conversion</li>
                        <li><strong>Input Data:</strong> Paste your JSON or data to process</li>
                        <li><strong>Set Options:</strong> Configure processing parameters</li>
                        <li><strong>Process Data:</strong> Click to perform the operation</li>
                        <li><strong>Review Results:</strong> Examine the output and validation status</li>
                        <li><strong>Copy or Download:</strong> Use the results as needed</li>
                        <li><strong>Chain Operations:</strong> Use results in other operations</li>
                    </ol>
                </article>

                <article class="content-card">
                    <h2>Use Cases for GoHighLevel Agencies</h2>
                    <p>This tool serves multiple purposes in your daily GoHighLevel operations:</p>
                    
                    <ul>
                        <li><strong>API Integration:</strong> Format and validate API responses</li>
                        <li><strong>Webhook Processing:</strong> Handle incoming webhook data</li>
                        <li><strong>Client Data Migration:</strong> Convert data between formats</li>
                        <li><strong>System Configuration:</strong> Format configuration files</li>
                        <li><strong>Data Export:</strong> Prepare data for client delivery</li>
                        <li><strong>Testing and Development:</strong> Verify JSON structure</li>
                        <li><strong>Client Support:</strong> Help clients with data issues</li>
                        <li><strong>Documentation:</strong> Create readable JSON examples</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>JSON Best Practices</h2>
                    <p>Following these best practices ensures optimal results:</p>
                    
                    <ul>
                        <li><strong>Proper Formatting:</strong> Use consistent indentation and spacing</li>
                        <li><strong>Key Naming:</strong> Use descriptive, consistent key names</li>
                        <li><strong>Data Types:</strong> Use appropriate data types for values</li>
                        <li><strong>Validation:</strong> Always validate JSON before use</li>
                        <li><strong>Error Handling:</strong> Implement proper error handling</li>
                        <li><strong>Performance:</strong> Consider JSON size for performance</li>
                        <li><strong>Security:</strong> Validate and sanitize JSON input</li>
                        <li><strong>Documentation:</strong> Document JSON structure and format</li>
                    </ul>
                </article>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="related-tools">
        <div class="container">
            <h2>Related GoHighLevel Tools</h2>
            <div class="tools-grid">
                <a href="base64-encoder-decoder.php" class="tool-link">
                    <div class="tool-icon">📝</div>
                    <h3>Base64 Encoder/Decoder</h3>
                    <p>Encode and decode data in Base64 format</p>
                </a>
                <a href="hash-generator.php" class="tool-link">
                    <div class="tool-icon">🔐</div>
                    <h3>Hash Generator</h3>
                    <p>Generate secure hashes for data verification</p>
                </a>
                <a href="text-case-converter.php" class="tool-link">
                    <div class="tool-icon">🔤</div>
                    <h3>Text Case Converter</h3>
                    <p>Convert text between different case formats</p>
                </a>
            </div>
        </div>
    </section>

    <?php inc('includes/footer.php'); ?>
    <script src="common.js"></script>
    <script>`n    `n    // Load footer`n    fetch("footer.php")`n        .then(response => response.text())`n        .then(data => {`n            document.getElementById("footer-placeholder").innerHTML = data;`n        });`n});`n`n
        // JSON Formatter & Validator functionality
        let currentTab = 'format';
        
        // Character counters
        document.getElementById('formatInput').addEventListener('input', function() {
            const count = this.value.length;
            document.getElementById('formatCharCounter').textContent = count;
        });
        
        document.getElementById('validateInput').addEventListener('input', function() {
            const count = this.value.length;
            document.getElementById('validateCharCounter').textContent = count;
        });
        
        document.getElementById('minifyInput').addEventListener('input', function() {
            const count = this.value.length;
            document.getElementById('minifyCharCounter').textContent = count;
        });
        
        document.getElementById('convertInput').addEventListener('input', function() {
            const count = this.value.length;
            document.getElementById('convertCharCounter').textContent = count;
        });
        
        function showTab(tabName) {
            const tabPanes = document.querySelectorAll('.tab-pane');
            tabPanes.forEach(pane => pane.classList.remove('active'));
            
            const tabButtons = document.querySelectorAll('.tab-btn');
            tabButtons.forEach(btn => btn.classList.remove('active'));
            
            document.getElementById(tabName + '-tab').classList.add('active');
            event.target.classList.add('active');
            currentTab = tabName;
        }
        
        function formatJSON() {
            const input = document.getElementById('formatInput').value.trim();
            const indent = document.getElementById('formatIndent').checked;
            const sortKeys = document.getElementById('formatSort').checked;
            const compactArrays = document.getElementById('formatCompact').checked;
            
            if (!input) {
                alert('Please enter some JSON to format');
                return;
            }
            
            try {
                let parsed = JSON.parse(input);
                
                if (sortKeys) {
                    parsed = sortObjectKeys(parsed);
                }
                
                let formatted;
                if (indent) {
                    formatted = JSON.stringify(parsed, null, 2);
                } else {
                    formatted = JSON.stringify(parsed);
                }
                
                if (compactArrays) {
                    formatted = formatted.replace(/\[\s*\n\s*/g, '[').replace(/\n\s*\]/g, ']');
                }
                
                displayFormatResult(input, formatted, true);
                
            } catch (error) {
                displayFormatResult(input, 'Invalid JSON: ' + error.message, false);
            }
        }
        
        function validateJSON() {
            const input = document.getElementById('validateInput').value.trim();
            
            if (!input) {
                alert('Please enter some JSON to validate');
                return;
            }
            
            try {
                const parsed = JSON.parse(input);
                const structure = analyzeJSONStructure(parsed);
                
                displayValidateResult(true, '', structure);
                
            } catch (error) {
                displayValidateResult(false, error.message, {});
            }
        }
        
        function minifyJSON() {
            const input = document.getElementById('minifyInput').value.trim();
            
            if (!input) {
                alert('Please enter some JSON to minify');
                return;
            }
            
            try {
                const parsed = JSON.parse(input);
                const minified = JSON.stringify(parsed);
                const originalSize = input.length;
                const minifiedSize = minified.length;
                const reduction = originalSize - minifiedSize;
                const ratio = ((reduction / originalSize) * 100).toFixed(1);
                
                displayMinifyResult(input, minified, reduction, ratio);
                
            } catch (error) {
                alert('Invalid JSON: ' + error.message);
            }
        }
        
        function convertToJSON() {
            const input = document.getElementById('convertInput').value.trim();
            const type = document.getElementById('convertType').value;
            
            if (!input) {
                alert('Please enter some data to convert');
                return;
            }
            
            try {
                let converted;
                let status = 'Successfully converted';
                
                switch (type) {
                    case 'csv':
                        converted = csvToJSON(input);
                        break;
                    case 'xml':
                        converted = xmlToJSON(input);
                        break;
                    case 'yaml':
                        converted = yamlToJSON(input);
                        break;
                    case 'query':
                        converted = queryToJSON(input);
                        break;
                    default:
                        throw new Error('Unsupported conversion type');
                }
                
                displayConvertResult(input, converted, status);
                
            } catch (error) {
                displayConvertResult(input, 'Conversion failed: ' + error.message, 'Conversion failed');
            }
        }
        
        function sortObjectKeys(obj) {
            if (typeof obj !== 'object' || obj === null) return obj;
            if (Array.isArray(obj)) return obj.map(sortObjectKeys);
            
            const sorted = {};
            Object.keys(obj).sort().forEach(key => {
                sorted[key] = sortObjectKeys(obj[key]);
            });
            return sorted;
        }
        
        function analyzeJSONStructure(obj) {
            const analysis = {
                type: typeof obj,
                isArray: Array.isArray(obj),
                size: obj === null ? 0 : Object.keys(obj).length,
                depth: getObjectDepth(obj),
                hasNested: hasNestedObjects(obj)
            };
            
            if (Array.isArray(obj)) {
                analysis.arrayLength = obj.length;
                analysis.arrayTypes = [...new Set(obj.map(item => typeof item))];
            }
            
            return analysis;
        }
        
        function getObjectDepth(obj, depth = 0) {
            if (typeof obj !== 'object' || obj === null) return depth;
            if (Array.isArray(obj)) {
                return Math.max(depth, ...obj.map(item => getObjectDepth(item, depth + 1)));
            }
            
            let maxDepth = depth;
            for (const key in obj) {
                maxDepth = Math.max(maxDepth, getObjectDepth(obj[key], depth + 1));
            }
            return maxDepth;
        }
        
        function hasNestedObjects(obj) {
            if (typeof obj !== 'object' || obj === null) return false;
            if (Array.isArray(obj)) {
                return obj.some(item => hasNestedObjects(item));
            }
            
            for (const key in obj) {
                if (typeof obj[key] === 'object' && obj[key] !== null) return true;
            }
            return false;
        }
        
        function csvToJSON(csv) {
            const lines = csv.trim().split('\n');
            const headers = lines[0].split(',').map(h => h.trim());
            const result = [];
            
            for (let i = 1; i < lines.length; i++) {
                const obj = {};
                const values = lines[i].split(',');
                
                headers.forEach((header, index) => {
                    obj[header] = values[index] ? values[index].trim() : '';
                });
                
                result.push(obj);
            }
            
            return result;
        }
        
        function xmlToJSON(xml) {
            // Simple XML to JSON conversion
            const parser = new DOMParser();
            const xmlDoc = parser.parseFromString(xml, 'text/xml');
            return xmlToJson(xmlDoc.documentElement);
        }
        
        function xmlToJson(xml) {
            let obj = {};
            
            if (xml.nodeType === 1) {
                if (xml.attributes.length > 0) {
                    obj['@attributes'] = {};
                    for (let j = 0; j < xml.attributes.length; j++) {
                        const attribute = xml.attributes.item(j);
                        obj['@attributes'][attribute.nodeName] = attribute.nodeValue;
                    }
                }
            } else if (xml.nodeType === 3) {
                obj = xml.nodeValue;
            }
            
            if (xml.hasChildNodes()) {
                for (let i = 0; i < xml.childNodes.length; i++) {
                    const item = xml.childNodes.item(i);
                    const nodeName = item.nodeName;
                    
                    if (typeof obj[nodeName] === 'undefined') {
                        obj[nodeName] = xmlToJson(item);
                    } else {
                        if (typeof obj[nodeName].push === 'undefined') {
                            const old = obj[nodeName];
                            obj[nodeName] = [];
                            obj[nodeName].push(old);
                        }
                        obj[nodeName].push(xmlToJson(item));
                    }
                }
            }
            return obj;
        }
        
        function yamlToJSON(yaml) {
            // Simple YAML to JSON conversion
            const lines = yaml.trim().split('\n');
            const result = {};
            let currentLevel = 0;
            let currentPath = [];
            
            for (const line of lines) {
                if (line.trim() === '') continue;
                
                const indent = line.search(/\S/);
                const content = line.trim();
                
                if (content.startsWith('-')) {
                    // Array item
                    if (!result[currentPath.join('.')]) {
                        result[currentPath.join('.')] = [];
                    }
                    result[currentPath.join('.')].push(content.substring(1).trim());
                } else if (content.includes(':')) {
                    // Key-value pair
                    const [key, value] = content.split(':', 2);
                    const fullKey = [...currentPath, key.trim()].join('.');
                    result[fullKey] = value ? value.trim() : '';
                }
            }
            
            return result;
        }
        
        function queryToJSON(query) {
            const result = {};
            const params = new URLSearchParams(query);
            
            for (const [key, value] of params) {
                if (result[key]) {
                    if (Array.isArray(result[key])) {
                        result[key].push(value);
                    } else {
                        result[key] = [result[key], value];
                    }
                } else {
                    result[key] = value;
                }
            }
            
            return result;
        }
        
        function displayFormatResult(input, formatted, isValid) {
            document.getElementById('formattedJSON').textContent = formatted;
            document.getElementById('validationStatus').textContent = isValid ? 'Valid JSON' : 'Invalid JSON';
            document.getElementById('outputSize').textContent = formatted.length + ' characters';
            document.getElementById('formatResult').style.display = 'block';
        }
        
        function displayValidateResult(isValid, error, structure) {
            document.getElementById('validateStatus').textContent = isValid ? '✅ Valid JSON' : '❌ Invalid JSON';
            document.getElementById('errorDetails').textContent = isValid ? 'No errors found' : error;
            document.getElementById('jsonStructure').textContent = isValid ? 
                `Type: ${structure.type}, Size: ${structure.size}, Depth: ${structure.depth}` : 'N/A';
            document.getElementById('validateResult').style.display = 'block';
        }
        
        function displayMinifyResult(input, minified, reduction, ratio) {
            document.getElementById('minifiedJSON').textContent = minified;
            document.getElementById('sizeReduction').textContent = reduction + ' characters saved';
            document.getElementById('compressionRatio').textContent = ratio + '% reduction';
            document.getElementById('minifyResult').style.display = 'block';
        }
        
        function displayConvertResult(input, converted, status) {
            document.getElementById('convertedJSON').textContent = JSON.stringify(converted, null, 2);
            document.getElementById('conversionStatus').textContent = status;
            document.getElementById('convertResult').style.display = 'block';
        }
        
        function clearFormat() {
            document.getElementById('formatInput').value = '';
            document.getElementById('formatResult').style.display = 'none';
            document.getElementById('formatCharCounter').textContent = '0';
        }
        
        function clearValidate() {
            document.getElementById('validateInput').value = '';
            document.getElementById('validateResult').style.display = 'none';
            document.getElementById('validateCharCounter').textContent = '0';
        }
        
        function clearMinify() {
            document.getElementById('minifyInput').value = '';
            document.getElementById('minifyResult').style.display = 'none';
            document.getElementById('minifyCharCounter').textContent = '0';
        }
        
        function clearConvert() {
            document.getElementById('convertInput').value = '';
            document.getElementById('convertResult').style.display = 'none';
            document.getElementById('convertCharCounter').textContent = '0';
        }
        
        function copyResult(elementId) {
            const text = document.getElementById(elementId).textContent;
            navigator.clipboard.writeText(text).then(() => {
                alert('Result copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy: ', err);
                const textArea = document.createElement('textarea');
                textArea.value = text;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('Result copied to clipboard!');
            });
        }
        
        function downloadFormatted() {
            const formatted = document.getElementById('formattedJSON').textContent;
            downloadFile(formatted, 'formatted-json.json', 'application/json');
        }
        
        function downloadMinified() {
            const minified = document.getElementById('minifiedJSON').textContent;
            downloadFile(minified, 'minified-json.json', 'application/json');
        }
        
        function downloadConverted() {
            const converted = document.getElementById('convertedJSON').textContent;
            downloadFile(converted, 'converted-json.json', 'application/json');
        }
        
        function downloadFile(content, filename, mimeType) {
            const blob = new Blob([content], { type: mimeType });
            const url = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            link.click();
            URL.revokeObjectURL(url);
        }
        
        function minifyFormatted() {
            const formatted = document.getElementById('formattedJSON').textContent;
            if (formatted) {
                document.getElementById('minifyInput').value = formatted;
                showTab('minify');
                minifyJSON();
            }
        }
        
        function formatValidated() {
            const input = document.getElementById('validateInput').value.trim();
            if (input) {
                document.getElementById('formatInput').value = input;
                showTab('format');
                formatJSON();
            }
        }
        
        function minifyFormatted() {
            const formatted = document.getElementById('formattedJSON').textContent;
            if (formatted) {
                document.getElementById('minifyInput').value = formatted;
                showTab('minify');
                minifyJSON();
            }
        }
        
        function formatMinified() {
            const minified = document.getElementById('minifiedJSON').textContent;
            if (minified) {
                document.getElementById('formatInput').value = minified;
                showTab('format');
                formatJSON();
            }
        }
        
        function formatConverted() {
            const converted = document.getElementById('convertedJSON').textContent;
            if (converted) {
                document.getElementById('formatInput').value = converted;
                showTab('format');
                formatJSON();
            }
        }
        
        function fixErrors() {
            // Basic error fixing functionality
            const input = document.getElementById('validateInput').value.trim();
            let fixed = input;
            
            // Fix common JSON errors
            fixed = fixed.replace(/(\w+):/g, '"$1":'); // Add quotes to unquoted keys
            fixed = fixed.replace(/'/g, '"'); // Replace single quotes with double quotes
            fixed = fixed.replace(/,(\s*[}\]])/g, '$1'); // Remove trailing commas
            
            try {
                JSON.parse(fixed);
                document.getElementById('validateInput').value = fixed;
                validateJSON();
            } catch (error) {
                alert('Unable to automatically fix errors. Please fix manually.');
            }
        }
    </script>

    <style>
        .json-tabs {
            display: flex;
            border-bottom: 2px solid #e9ecef;
            margin-bottom: 20px;
        }
        
        .tab-btn {
            background: none;
            border: none;
            padding: 10px 20px;
            cursor: pointer;
            border-bottom: 2px solid transparent;
            transition: all 0.3s ease;
        }
        
        .tab-btn.active {
            border-bottom-color: #007bff;
            color: #007bff;
            font-weight: 600;
        }
        
        .tab-btn:hover {
            background-color: #f8f9fa;
        }
        
        .tab-pane {
            display: none;
        }
        
        .tab-pane.active {
            display: block;
        }
        
        .char-counter {
            font-size: 12px;
            color: #666;
            text-align: right;
            margin-top: 5px;
        }
        
        .options-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 10px;
        }
        
        .checkbox-label {
            display: flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .checkbox-label input[type="checkbox"] {
            margin: 0;
        }
        
        .result-display {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .result-item {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            padding: 10px;
            background: white;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }
        
        .result-item label {
            font-weight: 600;
            min-width: 120px;
            color: #333;
        }
        
        .result-value {
            flex: 1;
            background: #f8f9fa;
            padding: 8px 12px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
            word-break: break-all;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .result-value pre {
            margin: 0;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            line-height: 1.4;
            white-space: pre-wrap;
        }
        
        .result-output {
            display: flex;
            align-items: flex-start;
            justify-content: space-between;
            gap: 10px;
        }
        
        .copy-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 4px 8px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            white-space: nowrap;
            margin-top: 8px;
        }
        
        .copy-btn:hover {
            background: #0056b3;
        }
        
        .result-actions {
            margin-top: 20px;
            text-align: center;
        }
        
        .result-actions .btn {
            margin: 0 10px;
        }
        
        @media (max-width: 768px) {
            .options-grid {
                grid-template-columns: 1fr;
            }
            
            .result-item {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .result-item label {
                min-width: auto;
                margin-bottom: 5px;
            }
            
            .json-tabs {
                flex-direction: column;
            }
            
            .tab-btn {
                text-align: left;
                border-bottom: none;
                border-left: 3px solid transparent;
            }
            
            .tab-btn.active {
                border-left-color: #007bff;
                border-bottom-color: transparent;
            }
        }
    </style>
</body>
</html>




