<?php
/**
 * SEO Helper Functions for GHL Toolset
 * Handles meta tags, structured data, and canonical URLs
 */

/**
 * Render meta tags for a page
 */
function render_meta($title, $description, $canonical, $keywords = '', $image = '') {
    $title = htmlspecialchars($title, ENT_QUOTES, 'UTF-8');
    $description = htmlspecialchars($description, ENT_QUOTES, 'UTF-8');
    $canonical = htmlspecialchars($canonical, ENT_QUOTES, 'UTF-8');
    $keywords = htmlspecialchars($keywords, ENT_QUOTES, 'UTF-8');
    $image = htmlspecialchars($image ?: BASE_URL . '/public/images/og-default.jpg', ENT_QUOTES, 'UTF-8');
    
    echo "<title>{$title}</title>\n";
    echo "<meta name=\"description\" content=\"{$description}\">\n";
    if ($keywords) {
        echo "<meta name=\"keywords\" content=\"{$keywords}\">\n";
    }
    echo "<link rel=\"canonical\" href=\"{$canonical}\">\n";
    
    // Open Graph
    echo "<meta property=\"og:title\" content=\"{$title}\">\n";
    echo "<meta property=\"og:description\" content=\"{$description}\">\n";
    echo "<meta property=\"og:url\" content=\"{$canonical}\">\n";
    echo "<meta property=\"og:image\" content=\"{$image}\">\n";
    echo "<meta property=\"og:type\" content=\"website\">\n";
    echo "<meta property=\"og:site_name\" content=\"" . SITE_NAME . "\">\n";
    
    // Twitter
    echo "<meta name=\"twitter:card\" content=\"summary_large_image\">\n";
    echo "<meta name=\"twitter:title\" content=\"{$title}\">\n";
    echo "<meta name=\"twitter:description\" content=\"{$description}\">\n";
    echo "<meta name=\"twitter:image\" content=\"{$image}\">\n";
}

/**
 * Render structured data (JSON-LD)
 */
function render_schema($jsonLd) {
    $json = json_encode($jsonLd, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if ($json === false) {
        error_log('Failed to encode JSON-LD schema');
        return;
    }
    echo "<script type=\"application/ld+json\">{$json}</script>\n";
}

/**
 * Generate SoftwareApplication schema for tools
 */
function get_tool_schema($toolName, $description, $url, $features = []) {
    return [
        '@context' => 'https://schema.org',
        '@type' => 'SoftwareApplication',
        'name' => $toolName,
        'description' => $description,
        'url' => $url,
        'applicationCategory' => 'BusinessApplication',
        'operatingSystem' => 'Web Browser',
        'browserRequirements' => 'Requires JavaScript. Requires HTML5.',
        'softwareVersion' => '1.0',
        'offers' => [
            '@type' => 'Offer',
            'price' => '0',
            'priceCurrency' => 'USD',
            'availability' => 'https://schema.org/InStock'
        ],
        'provider' => [
            '@type' => 'Organization',
            'name' => 'Paddle CRM',
            'description' => 'WhiteLabel GoHighLevel Agency',
            'url' => BASE_URL
        ],
        'featureList' => $features,
        'inLanguage' => 'en-US',
        'isAccessibleForFree' => true,
        'applicationSubCategory' => 'Marketing Tool'
    ];
}

/**
 * Generate FAQPage schema
 */
function get_faq_schema($faqs) {
    $questions = [];
    foreach ($faqs as $faq) {
        $questions[] = [
            '@type' => 'Question',
            'name' => $faq['question'],
            'acceptedAnswer' => [
                '@type' => 'Answer',
                'text' => $faq['answer']
            ]
        ];
    }
    
    return [
        '@context' => 'https://schema.org',
        '@type' => 'FAQPage',
        'mainEntity' => $questions
    ];
}

/**
 * Generate BreadcrumbList schema
 */
function get_breadcrumb_schema($breadcrumbs) {
    $items = [];
    foreach ($breadcrumbs as $index => $crumb) {
        $items[] = [
            '@type' => 'ListItem',
            'position' => $index + 1,
            'name' => $crumb['name'],
            'item' => $crumb['url']
        ];
    }
    
    return [
        '@context' => 'https://schema.org',
        '@type' => 'BreadcrumbList',
        'itemListElement' => $items
    ];
}

/**
 * Generate WebSite schema with search action
 */
function get_website_schema() {
    return [
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => SITE_NAME,
        'url' => BASE_URL,
        'description' => 'Free professional tools for GoHighLevel users and agencies',
        'potentialAction' => [
            '@type' => 'SearchAction',
            'target' => [
                '@type' => 'EntryPoint',
                'urlTemplate' => BASE_URL . '/?s={search_term_string}'
            ],
            'query-input' => 'required name=search_term_string'
        ],
        'publisher' => [
            '@type' => 'Organization',
            'name' => 'Paddle CRM',
            'url' => BASE_URL
        ]
    ];
}

/**
 * Generate Organization schema
 */
function get_organization_schema() {
    return [
        '@context' => 'https://schema.org',
        '@type' => 'Organization',
        'name' => 'Paddle CRM',
        'url' => BASE_URL,
        'description' => 'WhiteLabel GoHighLevel Agency providing professional tools and services',
        'foundingDate' => '2024',
        'areaServed' => 'Worldwide',
        'serviceType' => 'GoHighLevel Services',
        'knowsAbout' => [
            'GoHighLevel',
            'CRM Tools',
            'Marketing Automation',
            'Lead Generation',
            'SMS Marketing',
            'Email Marketing'
        ]
    ];
}

/**
 * Clean and optimize title for SEO
 */
function optimize_title($title, $suffix = '') {
    $title = trim($title);
    $suffix = $suffix ?: ' | ' . SITE_NAME;
    $maxLength = 60 - strlen($suffix);
    
    if (strlen($title) > $maxLength) {
        $title = substr($title, 0, $maxLength - 3) . '...';
    }
    
    return $title . $suffix;
}

/**
 * Clean and optimize description for SEO
 */
function optimize_description($description) {
    $description = trim(strip_tags($description));
    $maxLength = 160;
    
    if (strlen($description) > $maxLength) {
        $description = substr($description, 0, $maxLength - 3) . '...';
    }
    
    return $description;
}

/**
 * Generate canonical URL
 */
function get_canonical_url($path = '') {
    $path = ltrim($path, '/');
    return BASE_URL . ($path ? '/' . $path : '');
}

/**
 * Generate hreflang tags for international versions
 */
function render_hreflang($alternates = []) {
    foreach ($alternates as $lang => $url) {
        echo "<link rel=\"alternate\" hreflang=\"{$lang}\" href=\"{$url}\">\n";
    }
}

/**
 * Render preconnect links for performance
 */
function render_preconnects($domains = []) {
    $defaultDomains = [
        'https://fonts.googleapis.com',
        'https://fonts.gstatic.com'
    ];
    
    if (ENABLE_ANALYTICS && defined('__consent') && $_COOKIE['consent'] ?? false) {
        $defaultDomains[] = 'https://www.google-analytics.com';
        $defaultDomains[] = 'https://www.googletagmanager.com';
    }
    
    if (ENABLE_ADS && defined('__consent') && $_COOKIE['consent'] ?? false) {
        $defaultDomains[] = 'https://pagead2.googlesyndication.com';
        $defaultDomains[] = 'https://googleads.g.doubleclick.net';
    }
    
    $allDomains = array_unique(array_merge($defaultDomains, $domains));
    
    foreach ($allDomains as $domain) {
        echo "<link rel=\"preconnect\" href=\"{$domain}\">\n";
    }
}
?>