<?php
/**
 * Affiliate Management System for GHL Toolset
 * Centralized affiliate URLs, UTM tracking, and disclosure
 */

// Affiliate Partner Configuration
$AFFILIATES = [
    'twilio' => [
        'base' => 'https://www.twilio.com/try-twilio',
        'name' => 'Twilio',
        'description' => 'Programmable SMS and voice platform',
        'commission_rate' => '5%',
        'tools' => ['sms-counter', 'bulk-sms-sender'],
        'cta_text' => 'Get Twilio Credits'
    ],
    'mailgun' => [
        'base' => 'https://www.mailgun.com/email-api/',
        'name' => 'Mailgun',
        'description' => 'Email delivery service for developers',
        'commission_rate' => '10%',
        'tools' => ['email-deliverability-checker', 'email-template-builder'],
        'cta_text' => 'Try Mailgun Free'
    ],
    'sendinblue' => [
        'base' => 'https://www.brevo.com/',
        'name' => 'Brevo (formerly Sendinblue)',
        'description' => 'All-in-one marketing platform',
        'commission_rate' => '15%',
        'tools' => ['email-subject-line-scorer', 'email-template-builder'],
        'cta_text' => 'Start Free Trial'
    ],
    'cloudflare' => [
        'base' => 'https://www.cloudflare.com/',
        'name' => 'Cloudflare',
        'description' => 'Web performance and security',
        'commission_rate' => '20%',
        'tools' => ['page-speed-optimizer', 'security-scanner'],
        'cta_text' => 'Secure Your Site'
    ],
    'ahrefs' => [
        'base' => 'https://ahrefs.com/',
        'name' => 'Ahrefs',
        'description' => 'SEO tools for professionals',
        'commission_rate' => '20%',
        'tools' => ['keyword-density-checker', 'meta-tag-generator'],
        'cta_text' => 'Try Ahrefs'
    ],
    'cryptojs' => [
        'base' => 'https://cryptojs.gitbook.io/docs/',
        'name' => 'CryptoJS',
        'description' => 'JavaScript cryptography library',
        'commission_rate' => '0%',
        'tools' => ['hash-generator', 'password-generator'],
        'cta_text' => 'Learn More',
        'is_affiliate' => false
    ]
];

/**
 * Generate affiliate URL with UTM parameters
 */
function affiliate_url($partnerId, $overrides = []) {
    global $AFFILIATES;
    
    if (!isset($AFFILIATES[$partnerId])) {
        error_log("Unknown affiliate partner: {$partnerId}");
        return '#';
    }
    
    $partner = $AFFILIATES[$partnerId];
    $base = $partner['base'];
    
    // Get current tool context
    $currentTool = $overrides['tool'] ?? get_current_tool();
    
    // Build UTM parameters
    $utm = array_merge([
        'utm_source' => DEFAULT_UTM_SOURCE,
        'utm_medium' => DEFAULT_UTM_MEDIUM,
        'utm_campaign' => "affiliate-{$partnerId}",
        'utm_content' => $currentTool,
        'utm_term' => $overrides['term'] ?? ''
    ], $overrides['utm'] ?? []);
    
    // Add first-touch attribution if available
    $firstTouch = get_first_touch_utm();
    if ($firstTouch) {
        $utm['utm_source_original'] = $firstTouch['source'];
        $utm['utm_medium_original'] = $firstTouch['medium'];
        $utm['utm_campaign_original'] = $firstTouch['campaign'];
    }
    
    // Filter out empty values
    $utm = array_filter($utm);
    
    // Build final URL
    $separator = strpos($base, '?') === false ? '?' : '&';
    return $base . $separator . http_build_query($utm);
}

/**
 * Get current tool from URL or context
 */
function get_current_tool() {
    $script = basename($_SERVER['SCRIPT_NAME'], '.php');
    return $script === 'index' ? 'homepage' : $script;
}

/**
 * Store first-touch UTM parameters
 */
function store_first_touch_utm() {
    if (!isset($_COOKIE['first_touch_utm']) && 
        (isset($_GET['utm_source']) || isset($_GET['utm_medium']) || isset($_GET['utm_campaign']))) {
        
        $firstTouch = [
            'source' => $_GET['utm_source'] ?? '',
            'medium' => $_GET['utm_medium'] ?? '',
            'campaign' => $_GET['utm_campaign'] ?? '',
            'content' => $_GET['utm_content'] ?? '',
            'term' => $_GET['utm_term'] ?? '',
            'timestamp' => time()
        ];
        
        setcookie('first_touch_utm', json_encode($firstTouch), time() + (30 * 24 * 60 * 60), '/'); // 30 days
    }
}

/**
 * Get first-touch UTM parameters
 */
function get_first_touch_utm() {
    if (isset($_COOKIE['first_touch_utm'])) {
        return json_decode($_COOKIE['first_touch_utm'], true);
    }
    return null;
}

/**
 * Check if partner is an affiliate
 */
function is_affiliate($partnerId) {
    global $AFFILIATES;
    return isset($AFFILIATES[$partnerId]) && 
           ($AFFILIATES[$partnerId]['is_affiliate'] ?? true);
}

/**
 * Get affiliate disclosure text
 */
function get_affiliate_disclosure($partnerId = null) {
    if ($partnerId && !is_affiliate($partnerId)) {
        return '';
    }
    
    return '(Affiliate)';
}

/**
 * Render affiliate link with proper attributes
 */
function render_affiliate_link($partnerId, $options = []) {
    global $AFFILIATES;
    
    if (!isset($AFFILIATES[$partnerId])) {
        return '';
    }
    
    $partner = $AFFILIATES[$partnerId];
    $url = affiliate_url($partnerId, $options);
    $text = $options['text'] ?? $partner['cta_text'];
    $classes = $options['classes'] ?? 'btn btn-primary affiliate-link';
    $disclosure = is_affiliate($partnerId) ? ' ' . get_affiliate_disclosure($partnerId) : '';
    
    $attributes = [
        'href' => $url,
        'class' => $classes,
        'target' => '_blank',
        'rel' => is_affiliate($partnerId) ? 'nofollow sponsored' : 'nofollow',
        'data-partner' => $partnerId,
        'onclick' => "track('click_affiliate', {partner: '{$partnerId}', tool: '" . get_current_tool() . "'})"
    ];
    
    $attrString = '';
    foreach ($attributes as $key => $value) {
        $attrString .= ' ' . $key . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
    }
    
    return "<a{$attrString}>" . htmlspecialchars($text) . $disclosure . "</a>";
}

/**
 * Get relevant affiliate partners for a tool
 */
function get_tool_affiliates($toolName) {
    global $AFFILIATES;
    $relevant = [];
    
    foreach ($AFFILIATES as $id => $partner) {
        if (in_array($toolName, $partner['tools'] ?? [])) {
            $relevant[$id] = $partner;
        }
    }
    
    return $relevant;
}

/**
 * Render affiliate disclosure notice
 */
function render_affiliate_disclosure_notice() {
    echo '<div class="affiliate-disclosure">';
    echo '<p><small>';
    echo 'Some links on this page are affiliate links, which means we earn a small commission ';
    echo 'if you purchase through them. This helps us maintain these free tools. ';
    echo '<a href="/legal/affiliate-disclosure.php">Learn more</a>';
    echo '</small></p>';
    echo '</div>';
}

/**
 * Track affiliate click for analytics
 */
function track_affiliate_click($partnerId, $tool, $placement = 'content') {
    // This will be called via JavaScript onclick events
    // Data is passed to analytics.js track() function
    return [
        'event' => 'click_affiliate',
        'partner_id' => $partnerId,
        'tool' => $tool,
        'placement' => $placement,
        'timestamp' => time()
    ];
}

/**
 * Get affiliate performance data (for internal tracking)
 */
function get_affiliate_stats($partnerId, $period = 30) {
    // This would integrate with analytics to track:
    // - Click-through rates
    // - Conversion tracking
    // - Revenue attribution
    // For now, return placeholder structure
    return [
        'clicks' => 0,
        'conversions' => 0,
        'revenue' => 0.00,
        'ctr' => 0.00,
        'conversion_rate' => 0.00
    ];
}

// Initialize first-touch tracking
store_first_touch_utm();
?>