﻿<?php
require_once __DIR__ . '/config.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hash Generator for GoHighLevel | Free Hash Generation Tool</title>
    <meta name="description" content="Free hash generator for GoHighLevel users. Generate MD5, SHA-1, SHA-256, and other hash algorithms. Perfect for agencies needing secure data handling and verification.">
    <meta name="keywords" content="hash generator, MD5 generator, SHA generator, hash calculator, data security, GoHighLevel agency, password hashing">
    <meta name="author" content="Paddle CRM">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph / Facebook -->
    <meta property="og:type" content="website">
    <meta property="og:title" content="Hash Generator for GoHighLevel">
    <meta property="og:description" content="Free hash generator for GoHighLevel users. Generate MD5, SHA-1, SHA-256, and other hash algorithms.">
    <meta property="og:url" content="https://yourdomain.com/hash-generator.php">
    <meta property="og:image" content="https://yourdomain.com/images/hash-generator-og.jpg">
    
    <!-- Twitter -->
    <meta property="twitter:card" content="summary_large_image">
    <meta property="twitter:title" content="Hash Generator for GoHighLevel">
    <meta property="twitter:description" content="Free hash generator for GoHighLevel users. Generate MD5, SHA-1, SHA-256, and other hash algorithms.">
    <meta property="twitter:image" content="https://yourdomain.com/images/hash-generator-og.jpg">
    
    <!-- Schema.org structured data -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "Hash Generator for GoHighLevel",
        "description": "Free tool to generate various hash algorithms for data security and verification",
        "url": "https://yourdomain.com/hash-generator.php",
        "applicationCategory": "BusinessApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Paddle CRM",
            "description": "WhiteLabel GoHighLevel Agency"
        }
    }
    </script>
    
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <?php inc('includes/header.php'); ?>

    <!-- Hero Section -->
    <section class="hero tool-hero">
        <div class="container">
            <nav class="breadcrumb">
                <a href="index.php">Home</a> > <span>Hash Generator</span>
            </nav>
            <div class="hero-content">
                <div class="hero-text">
                    <h1 class="hero-title">Hash Generator for GoHighLevel</h1>
                    <p class="hero-subtitle">Generate secure hashes using MD5, SHA-1, SHA-256, and other algorithms for data verification and security</p>
                    <div class="hero-stats">
                        <div class="stat-item">
                            <span class="stat-number">Professional</span>
                            <span class="stat-label">Quality</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">GoHighLevel</span>
                            <span class="stat-label">Optimized</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">100%</span>
                            <span class="stat-label">Free Tool</span>
                        </div>
                    </div>
                </div>
                <div class="hero-visual">
                    <div class="hero-image">
                        <div class="floating-card card-1">
                            <span class="card-icon">🔐</span>
                            <span class="card-text">Hash</span>
                        </div>
                        <div class="floating-card card-2">
                            <span class="card-icon">⚡</span>
                            <span class="card-text">Fast</span>
                        </div>
                        <div class="floating-card card-3">
                            <span class="card-icon">🎯</span>
                            <span class="card-text">Secure</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Section -->
    <section class="tool-section">
        <div class="container">
            <div class="tool-card featured">
                <div class="tool-header">
                    <div class="tool-icon">🔐</div>
                    <h2>Hash Generator</h2>
                    <p>Professional hash generation tool designed specifically for GoHighLevel users and agencies</p>
                </div>
                
                <div class="tool-interface">
                    <div class="input-group">
                        <label for="inputText">Input Text or Data</label>
                        <textarea id="inputText" rows="6" placeholder="Enter the text, password, or data you want to hash..."></textarea>
                        <div class="char-counter">
                            <span id="charCounter">0</span> characters
                        </div>
                    </div>
                    
                    <div class="input-group">
                        <label for="hashAlgorithm">Hash Algorithm</label>
                        <select id="hashAlgorithm">
                            <option value="md5">MD5 (128-bit)</option>
                            <option value="sha1">SHA-1 (160-bit)</option>
                            <option value="sha256">SHA-256 (256-bit)</option>
                            <option value="sha512">SHA-512 (512-bit)</option>
                            <option value="sha224">SHA-224 (224-bit)</option>
                            <option value="sha384">SHA-384 (384-bit)</option>
                            <option value="ripemd160">RIPEMD-160 (160-bit)</option>
                            <option value="whirlpool">Whirlpool (512-bit)</option>
                        </select>
                    </div>
                    
                    <div class="input-group">
                        <label for="hashFormat">Output Format</label>
                        <select id="hashFormat">
                            <option value="hex">Hexadecimal (default)</option>
                            <option value="base64">Base64</option>
                            <option value="binary">Binary</option>
                        </select>
                    </div>
                    
                    <div class="input-group">
                        <label for="saltValue">Salt (Optional)</label>
                        <input type="text" id="saltValue" placeholder="Enter salt value for additional security">
                        <div class="salt-info">
                            <small>Adding salt increases security by making hash cracking more difficult</small>
                        </div>
                    </div>
                    
                    <div class="button-group">
                        <button class="btn btn-primary" onclick="generateHash()">Generate Hash</button>
                        <button class="btn btn-secondary" onclick="verifyHash()">Verify Hash</button>
                        <button class="btn btn-secondary" onclick="clearResults()">Clear Results</button>
                    </div>
                    
                    <div class="result" id="hashResult" style="display: none;">
                        <h3>Generated Hash Results</h3>
                        
                        <div class="hash-display">
                            <div class="hash-item">
                                <label>Input Text:</label>
                                <div class="hash-value" id="displayInput"></div>
                            </div>
                            
                            <div class="hash-item">
                                <label>Algorithm:</label>
                                <div class="hash-value" id="displayAlgorithm"></div>
                            </div>
                            
                            <div class="hash-item">
                                <label>Generated Hash:</label>
                                <div class="hash-value hash-output" id="displayHash">
                                    <span id="hashText"></span>
                                    <button class="copy-btn" onclick="copyHash()">📋 Copy</button>
                                </div>
                            </div>
                            
                            <div class="hash-item" id="saltDisplay" style="display: none;">
                                <label>Salt Used:</label>
                                <div class="hash-value" id="displaySalt"></div>
                            </div>
                            
                            <div class="hash-item">
                                <label>Hash Length:</label>
                                <div class="hash-value" id="displayLength"></div>
                            </div>
                        </div>
                        
                        <div class="hash-actions">
                            <button class="btn btn-primary" onclick="downloadHash()">Download Hash</button>
                            <button class="btn btn-secondary" onclick="generateMultipleHashes()">Generate All Algorithms</button>
                            <button class="btn btn-secondary" onclick="showHashInfo()">Hash Information</button>
                        </div>
                        
                        <div class="hash-comparison" id="hashComparison" style="display: none;">
                            <h4>All Hash Algorithms</h4>
                            <div class="comparison-grid" id="comparisonGrid"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Content Section -->
    <section class="content-section">
        <div class="container">
            <div class="content-grid">
                <article class="content-card">
                    <h2>What are Hash Functions?</h2>
                    <p>Hash functions are mathematical algorithms that take input data of any size and produce a fixed-size string of characters, which is typically a sequence of numbers and letters. The output, called a hash value or digest, is unique to the input data and serves as a digital fingerprint.</p>
                    
                    <p>For GoHighLevel users, understanding hash functions is crucial because:</p>
                    <ul>
                        <li><strong>Data Security:</strong> Hash functions protect sensitive information like passwords and API keys</li>
                        <li><strong>Data Integrity:</strong> Verify that data hasn't been tampered with during transmission</li>
                        <li><strong>Digital Signatures:</strong> Create and verify digital signatures for documents</li>
                        <li><strong>Password Storage:</strong> Securely store user passwords without revealing the actual passwords</li>
                        <li><strong>File Verification:</strong> Ensure downloaded files are authentic and unchanged</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Why GoHighLevel Users Need This Tool</h2>
                    <p>As a GoHighLevel agency, you're dealing with sensitive client data, API integrations, and security requirements. Hash functions are essential tools for maintaining data security and integrity in your operations.</p>
                    
                    <ul>
                        <li><strong>Client Data Protection:</strong> Secure sensitive client information and credentials</li>
                        <li><strong>API Security:</strong> Generate secure API keys and verify API requests</li>
                        <li><strong>Password Management:</strong> Create secure password hashes for client systems</li>
                        <li><strong>Data Verification:</strong> Verify data integrity in client communications</li>
                        <li><strong>Compliance Requirements:</strong> Meet security standards for client data handling</li>
                        <li><strong>Technical Expertise:</strong> Demonstrate security knowledge to clients</li>
                        <li><strong>Integration Security:</strong> Secure integrations with third-party services</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>How to Use the Hash Generator</h2>
                    <p>Our hash generator is designed to provide comprehensive hashing capabilities with professional results:</p>
                    
                    <ol>
                        <li><strong>Input Your Data:</strong> Enter the text, password, or data you want to hash</li>
                        <li><strong>Select Algorithm:</strong> Choose from MD5, SHA-1, SHA-256, SHA-512, and more</li>
                        <li><strong>Choose Format:</strong> Select hexadecimal, Base64, or binary output format</li>
                        <li><strong>Add Salt (Optional):</strong> Include salt for additional security</li>
                        <li><strong>Generate Hash:</strong> Click to create the hash value</li>
                        <li><strong>Review Results:</strong> Examine the generated hash and metadata</li>
                        <li><strong>Copy or Download:</strong> Use the hash value as needed</li>
                    </ol>
                </article>

                <article class="content-card">
                    <h2>Use Cases for GoHighLevel Agencies</h2>
                    <p>This tool serves multiple purposes in your daily GoHighLevel operations:</p>
                    
                    <ul>
                        <li><strong>Password Security:</strong> Hash client passwords before storage</li>
                        <li><strong>API Key Generation:</strong> Create secure API keys for integrations</li>
                        <li><strong>Data Verification:</strong> Verify data integrity in client communications</li>
                        <li><strong>File Integrity:</strong> Check if files have been modified</li>
                        <li><strong>Digital Signatures:</strong> Create signatures for important documents</li>
                        <li><strong>Security Audits:</strong> Verify security implementations for clients</li>
                        <li><strong>Compliance Testing:</strong> Ensure security measures meet standards</li>
                        <li><strong>Client Education:</strong> Teach clients about data security</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Hash Algorithm Types</h2>
                    <p>Understanding these hash algorithm categories helps you choose the right one for your needs:</p>
                    
                    <ul>
                        <li><strong>MD5 (Message Digest 5):</strong> 128-bit hash, fast but cryptographically broken</li>
                        <li><strong>SHA-1 (Secure Hash Algorithm 1):</strong> 160-bit hash, deprecated for security</li>
                        <li><strong>SHA-256 (Secure Hash Algorithm 256):</strong> 256-bit hash, current standard for security</li>
                        <li><strong>SHA-512 (Secure Hash Algorithm 512):</strong> 512-bit hash, highest security level</li>
                        <li><strong>RIPEMD-160:</strong> 160-bit hash, alternative to SHA-1</li>
                        <li><strong>Whirlpool:</strong> 512-bit hash, European standard</li>
                        <li><strong>BLAKE2:</strong> Modern hash function with high performance</li>
                        <li><strong>Argon2:</strong> Password hashing algorithm for secure storage</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Integration with GoHighLevel</h2>
                    <p>This tool integrates seamlessly with your GoHighLevel workflow:</p>
                    
                    <ul>
                        <li><strong>Client Onboarding:</strong> Generate secure credentials during setup</li>
                        <li><strong>API Integration:</strong> Create secure API keys for services</li>
                        <li><strong>Data Security:</strong> Hash sensitive client information</li>
                        <li><strong>Password Management:</strong> Secure password storage systems</li>
                        <li><strong>File Security:</strong> Verify file integrity for clients</li>
                        <li><strong>Compliance Audits:</strong> Demonstrate security measures</li>
                        <li><strong>Client Training:</strong> Educate clients about security best practices</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Hash Security Best Practices</h2>
                    <p>Following these best practices ensures optimal security:</p>
                    
                    <ul>
                        <li><strong>Algorithm Selection:</strong> Use SHA-256 or SHA-512 for security-critical applications</li>
                        <li><strong>Salt Implementation:</strong> Always add unique salt to password hashes</li>
                        <li><strong>Hash Storage:</strong> Store only hash values, never plain text</li>
                        <li><strong>Regular Updates:</strong> Update hash algorithms as security standards evolve</li>
                        <li><strong>Verification Process:</strong> Implement proper hash verification procedures</li>
                        <li><strong>Key Management:</strong> Secure storage and rotation of hash keys</li>
                        <li><strong>Audit Logging:</strong> Log hash generation and verification activities</li>
                        <li><strong>Performance Testing:</strong> Test hash performance for large datasets</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Advanced Hash Strategies</h2>
                    <p>Beyond basic implementation, consider these advanced strategies:</p>
                    
                    <ul>
                        <li><strong>Key Stretching:</strong> Use multiple hash iterations for passwords</li>
                        <li><strong>Hash Chaining:</strong> Chain multiple hash functions for enhanced security</li>
                        <li><strong>Adaptive Hashing:</strong> Adjust hash strength based on data sensitivity</li>
                        <li><strong>Hash Verification:</strong> Implement automated hash checking systems</li>
                        <li><strong>Performance Optimization:</strong> Choose algorithms based on speed vs. security needs</li>
                        <li><strong>Multi-Algorithm Support:</strong> Support multiple hash algorithms for compatibility</li>
                        <li><strong>Hash Synchronization:</strong> Keep hash values synchronized across systems</li>
                        <li><strong>Security Monitoring:</strong> Monitor for hash collision attempts</li>
                    </ul>
                </article>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="related-tools">
        <div class="container">
            <h2>Related GoHighLevel Tools</h2>
            <div class="tools-grid">
                <a href="password-generator.php" class="tool-link">
                    <div class="tool-icon">🔑</div>
                    <h3>Password Generator</h3>
                    <p>Generate secure passwords for your accounts and systems</p>
                </a>
                <a href="base64-encoder-decoder.php" class="tool-link">
                    <div class="tool-icon">📝</div>
                    <h3>Base64 Encoder/Decoder</h3>
                    <p>Encode and decode data in Base64 format</p>
                </a>
                <a href="json-formatter-validator.php" class="tool-link">
                    <div class="tool-icon">📊</div>
                    <h3>JSON Formatter & Validator</h3>
                    <p>Format and validate JSON data for your applications</p>
                </a>
            </div>
        </div>
    </section>

    <?php inc('includes/footer.php'); ?>
    <script src="common.js"></script>
    <script>`n    `n    // Load footer`n    fetch("footer.php")`n        .then(response => response.text())`n        .then(data => {`n            document.getElementById("footer-placeholder").innerHTML = data;`n        });`n});`n`n
        // Hash Generator functionality
        let currentHash = '';
        let currentInput = '';
        
        // Character counter
        document.getElementById('inputText').addEventListener('input', function() {
            const count = this.value.length;
            document.getElementById('charCounter').textContent = count;
        });
        
        // CryptoJS library for hash generation
        function loadCryptoJS() {
            if (typeof CryptoJS === 'undefined') {
                const script = document.createElement('script');
                script.src = 'https://cdnjs.cloudflare.com/ajax/libs/crypto-js/4.1.1/crypto-js.min.js';
                script.onload = function() {
                    console.log('CryptoJS loaded successfully');
                };
                document.head.appendChild(script);
            }
        }
        
        // Load CryptoJS when page loads
        loadCryptoJS();
        
        function generateHash() {
            const input = document.getElementById('inputText').value.trim();
            const algorithm = document.getElementById('hashAlgorithm').value;
            const format = document.getElementById('hashFormat').value;
            const salt = document.getElementById('saltValue').value.trim();
            
            if (!input) {
                alert('Please enter some text to hash');
                return;
            }
            
            if (typeof CryptoJS === 'undefined') {
                alert('Hash library is still loading. Please wait a moment and try again.');
                return;
            }
            
            try {
                let hashValue = '';
                let saltedInput = input;
                
                if (salt) {
                    saltedInput = input + salt;
                }
                
                // Generate hash based on selected algorithm
                switch (algorithm) {
                    case 'md5':
                        hashValue = CryptoJS.MD5(saltedInput);
                        break;
                    case 'sha1':
                        hashValue = CryptoJS.SHA1(saltedInput);
                        break;
                    case 'sha256':
                        hashValue = CryptoJS.SHA256(saltedInput);
                        break;
                    case 'sha512':
                        hashValue = CryptoJS.SHA512(saltedInput);
                        break;
                    case 'sha224':
                        hashValue = CryptoJS.SHA224(saltedInput);
                        break;
                    case 'sha384':
                        hashValue = CryptoJS.SHA384(saltedInput);
                        break;
                    case 'ripemd160':
                        hashValue = CryptoJS.RIPEMD160(saltedInput);
                        break;
                    case 'whirlpool':
                        hashValue = CryptoJS.Whirlpool(saltedInput);
                        break;
                    default:
                        hashValue = CryptoJS.SHA256(saltedInput);
                }
                
                // Format output based on selection
                let formattedHash = '';
                switch (format) {
                    case 'hex':
                        formattedHash = hashValue.toString();
                        break;
                    case 'base64':
                        formattedHash = hashValue.toString(CryptoJS.enc.Base64);
                        break;
                    case 'binary':
                        formattedHash = hashValue.toString(CryptoJS.enc.Latin1);
                        break;
                    default:
                        formattedHash = hashValue.toString();
                }
                
                displayHashResult(input, algorithm, formattedHash, salt, formattedHash.length);
                
            } catch (error) {
                console.error('Hash generation error:', error);
                alert('Error generating hash. Please try again.');
            }
        }
        
        function displayHashResult(input, algorithm, hash, salt, length) {
            currentHash = hash;
            currentInput = input;
            
            document.getElementById('displayInput').textContent = input;
            document.getElementById('displayAlgorithm').textContent = algorithm.toUpperCase();
            document.getElementById('displayHash').innerHTML = `<span id="hashText">${hash}</span><button class="copy-btn" onclick="copyHash()">📋 Copy</button>`;
            document.getElementById('displayLength').textContent = length + ' characters';
            
            if (salt) {
                document.getElementById('saltDisplay').style.display = 'block';
                document.getElementById('displaySalt').textContent = salt;
            } else {
                document.getElementById('saltDisplay').style.display = 'none';
            }
            
            document.getElementById('hashResult').style.display = 'block';
            document.getElementById('hashComparison').style.display = 'none';
        }
        
        function verifyHash() {
            const input = document.getElementById('inputText').value.trim();
            const algorithm = document.getElementById('hashAlgorithm').value;
            const salt = document.getElementById('saltValue').value.trim();
            
            if (!input || !currentHash) {
                alert('Please generate a hash first, then enter text to verify');
                return;
            }
            
            try {
                let saltedInput = input;
                if (salt) {
                    saltedInput = input + salt;
                }
                
                let newHash = '';
                switch (algorithm) {
                    case 'md5':
                        newHash = CryptoJS.MD5(saltedInput).toString();
                        break;
                    case 'sha1':
                        newHash = CryptoJS.SHA1(saltedInput).toString();
                        break;
                    case 'sha256':
                        newHash = CryptoJS.SHA256(saltedInput).toString();
                        break;
                    case 'sha512':
                        newHash = CryptoJS.SHA512(saltedInput).toString();
                        break;
                    case 'sha224':
                        newHash = CryptoJS.SHA224(saltedInput).toString();
                        break;
                    case 'sha384':
                        newHash = CryptoJS.SHA384(saltedInput).toString();
                        break;
                    case 'ripemd160':
                        newHash = CryptoJS.RIPEMD160(saltedInput).toString();
                        break;
                    case 'whirlpool':
                        newHash = CryptoJS.Whirlpool(saltedInput).toString();
                        break;
                }
                
                if (newHash === currentHash) {
                    alert('✅ Hash verification successful! The input matches the generated hash.');
                } else {
                    alert('❌ Hash verification failed! The input does not match the generated hash.');
                }
                
            } catch (error) {
                console.error('Hash verification error:', error);
                alert('Error verifying hash. Please try again.');
            }
        }
        
        function clearResults() {
            document.getElementById('inputText').value = '';
            document.getElementById('saltValue').value = '';
            document.getElementById('hashResult').style.display = 'none';
            document.getElementById('hashComparison').style.display = 'none';
            document.getElementById('charCounter').textContent = '0';
            currentHash = '';
            currentInput = '';
        }
        
        function copyHash() {
            const hashText = document.getElementById('hashText').textContent;
            navigator.clipboard.writeText(hashText).then(() => {
                alert('Hash copied to clipboard!');
            }).catch(err => {
                console.error('Failed to copy: ', err);
                // Fallback for older browsers
                const textArea = document.createElement('textarea');
                textArea.value = hashText;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('Hash copied to clipboard!');
            });
        }
        
        function downloadHash() {
            const input = document.getElementById('inputText').value.trim();
            const algorithm = document.getElementById('hashAlgorithm').value;
            const hash = currentHash;
            const salt = document.getElementById('saltValue').value.trim();
            
            const data = {
                input: input,
                algorithm: algorithm,
                hash: hash,
                salt: salt,
                timestamp: new Date().toISOString(),
                length: hash.length
            };
            
            const dataStr = JSON.stringify(data, null, 2);
            const dataBlob = new Blob([dataStr], {type: 'application/json'});
            
            const link = document.createElement('a');
            link.href = URL.createObjectURL(dataBlob);
            link.download = `hash-${algorithm}-${Date.now()}.json`;
            link.click();
        }
        
        function generateMultipleHashes() {
            const input = document.getElementById('inputText').value.trim();
            const salt = document.getElementById('saltValue').value.trim();
            
            if (!input) {
                alert('Please enter some text to hash');
                return;
            }
            
            if (typeof CryptoJS === 'undefined') {
                alert('Hash library is still loading. Please wait a moment and try again.');
                return;
            }
            
            try {
                const algorithms = ['md5', 'sha1', 'sha256', 'sha512', 'sha224', 'sha384', 'ripemd160', 'whirlpool'];
                const results = [];
                
                algorithms.forEach(alg => {
                    let saltedInput = input;
                    if (salt) {
                        saltedInput = input + salt;
                    }
                    
                    let hash = '';
                    switch (alg) {
                        case 'md5':
                            hash = CryptoJS.MD5(saltedInput).toString();
                            break;
                        case 'sha1':
                            hash = CryptoJS.SHA1(saltedInput).toString();
                            break;
                        case 'sha256':
                            hash = CryptoJS.SHA256(saltedInput).toString();
                            break;
                        case 'sha512':
                            hash = CryptoJS.SHA512(saltedInput).toString();
                            break;
                        case 'sha224':
                            hash = CryptoJS.SHA224(saltedInput).toString();
                            break;
                        case 'sha384':
                            hash = CryptoJS.SHA384(saltedInput).toString();
                            break;
                        case 'ripemd160':
                            hash = CryptoJS.RIPEMD160(saltedInput).toString();
                            break;
                        case 'whirlpool':
                            hash = CryptoJS.Whirlpool(saltedInput).toString();
                            break;
                    }
                    
                    results.push({
                        algorithm: alg.toUpperCase(),
                        hash: hash,
                        length: hash.length
                    });
                });
                
                displayComparisonGrid(results);
                
            } catch (error) {
                console.error('Multiple hash generation error:', error);
                alert('Error generating multiple hashes. Please try again.');
            }
        }
        
        function displayComparisonGrid(results) {
            const grid = document.getElementById('comparisonGrid');
            let html = '';
            
            results.forEach(result => {
                html += `
                    <div class="comparison-item">
                        <div class="comparison-algorithm">${result.algorithm}</div>
                        <div class="comparison-hash">${result.hash}</div>
                        <div class="comparison-length">${result.length} chars</div>
                    </div>
                `;
            });
            
            grid.innerHTML = html;
            document.getElementById('hashComparison').style.display = 'block';
        }
        
        function showHashInfo() {
            const algorithm = document.getElementById('hashAlgorithm').value;
            let info = '';
            
            switch (algorithm) {
                case 'md5':
                    info = 'MD5 (Message Digest 5) is a 128-bit hash function. It is cryptographically broken and should not be used for security purposes.';
                    break;
                case 'sha1':
                    info = 'SHA-1 (Secure Hash Algorithm 1) is a 160-bit hash function. It is deprecated for security use due to vulnerabilities.';
                    break;
                case 'sha256':
                    info = 'SHA-256 is a 256-bit hash function and is currently the standard for cryptographic security. It is widely used and considered secure.';
                    break;
                case 'sha512':
                    info = 'SHA-512 is a 512-bit hash function providing the highest level of security. It is slower than SHA-256 but more secure.';
                    break;
                case 'sha224':
                    info = 'SHA-224 is a 224-bit hash function derived from SHA-256. It provides good security with smaller output size.';
                    break;
                case 'sha384':
                    info = 'SHA-384 is a 384-bit hash function derived from SHA-512. It provides excellent security with reasonable performance.';
                    break;
                case 'ripemd160':
                    info = 'RIPEMD-160 is a 160-bit hash function designed as an alternative to SHA-1. It is considered secure.';
                    break;
                case 'whirlpool':
                    info = 'Whirlpool is a 512-bit hash function designed by Vincent Rijmen. It is a European standard and considered secure.';
                    break;
            }
            
            alert(`Hash Algorithm Information:\n\n${info}`);
        }
    </script>

    <style>
        .char-counter {
            font-size: 12px;
            color: #666;
            text-align: right;
            margin-top: 5px;
        }
        
        .salt-info {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
            font-style: italic;
        }
        
        .hash-display {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .hash-item {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            padding: 10px;
            background: white;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }
        
        .hash-item label {
            font-weight: 600;
            min-width: 120px;
            color: #333;
        }
        
        .hash-value {
            flex: 1;
            font-family: 'Courier New', monospace;
            background: #f8f9fa;
            padding: 8px 12px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
            word-break: break-all;
        }
        
        .hash-output {
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 10px;
        }
        
        .copy-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 4px 8px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            white-space: nowrap;
        }
        
        .copy-btn:hover {
            background: #0056b3;
        }
        
        .hash-actions {
            margin-top: 20px;
            text-align: center;
        }
        
        .hash-actions .btn {
            margin: 0 10px;
        }
        
        .hash-comparison {
            margin-top: 20px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
        }
        
        .comparison-grid {
            display: grid;
            gap: 15px;
            margin-top: 15px;
        }
        
        .comparison-item {
            display: grid;
            grid-template-columns: 120px 1fr 100px;
            gap: 15px;
            align-items: center;
            padding: 15px;
            background: white;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }
        
        .comparison-algorithm {
            font-weight: 600;
            color: #333;
        }
        
        .comparison-hash {
            font-family: 'Courier New', monospace;
            background: #f8f9fa;
            padding: 8px 12px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
            word-break: break-all;
            font-size: 12px;
        }
        
        .comparison-length {
            text-align: center;
            color: #666;
            font-size: 14px;
        }
        
        @media (max-width: 768px) {
            .comparison-item {
                grid-template-columns: 1fr;
                gap: 10px;
            }
            
            .hash-item {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .hash-item label {
                min-width: auto;
                margin-bottom: 5px;
            }
        }
    </style>
</body>
</html>




